/**
 * AcreOne - Background Service Worker
 * v2.1.0 - Security hardening (XSS, FETCH whitelist)
 */

// Установка расширения
chrome.runtime.onInstalled.addListener(async (details) => {
  // Получаем текущие настройки чтобы не сбрасывать токен при обновлении
  const currentSettings = await chrome.storage.local.get([
    'acreone_api_url',
    'acreone_token',
    'collect_size',
    'collect_article',
    'collect_color',
    'collect_buyer_name',
    'collect_photo',
    'allowed_shops',
    'max_shops'
  ]);

  // Дефолтные настройки (только если их нет)
  chrome.storage.local.set({
    acreone_api_url: currentSettings.acreone_api_url || 'https://api.acreone.ru',
    acreone_token: currentSettings.acreone_token || '',  // НЕ сбрасываем токен!
    // Настройки сбора (по умолчанию всё включено)
    collect_size: currentSettings.collect_size ?? true,
    collect_article: currentSettings.collect_article ?? true,
    collect_color: currentSettings.collect_color ?? true,
    collect_buyer_name: currentSettings.collect_buyer_name ?? true,
    collect_photo: currentSettings.collect_photo ?? true,
    // Информация о магазинах
    allowed_shops: currentSettings.allowed_shops || [],
    max_shops: currentSettings.max_shops || 1
  });

  // Контекстное меню (с защитой от ошибок)
  try {
    chrome.contextMenus.create({
      id: 'acreone-collect',
      title: 'Собрать данные заказа',
      contexts: ['page'],
      documentUrlPatterns: ['https://www.avito.ru/orders/*']
    });
  } catch (e) {
    // Контекстное меню уже существует
  }
});

// Обработка port connections (для долгих операций - держит SW активным)
chrome.runtime.onConnect.addListener((port) => {
  if (port.name === 'acreone-import') {
    port.onMessage.addListener(async (message) => {
      if (message.type === 'IMPORT_ORDERS') {
        try {
          const result = await importOrders(message.orders);
          port.postMessage(result);
        } catch (error) {
          console.error('[AcreOne BG] Ошибка:', error);
          port.postMessage({ success: false, error: error.message });
        }
      }
    });
  }
});

// Обработка сообщений от content scripts (для быстрых операций)
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'GET_SETTINGS') {
    chrome.storage.local.get([
      'acreone_api_url',
      'acreone_token',
      'collect_size',
      'collect_article',
      'collect_color',
      'collect_buyer_name',
      'collect_photo',
      'allowed_shops',
      'max_shops'
    ], (result) => {
      sendResponse(result);
    });
    return true;
  }

  if (message.type === 'SAVE_SETTINGS') {
    chrome.storage.local.set(message.settings, () => {
      sendResponse({ success: true });
    });
    return true;
  }

  // Загрузка настроек с сервера
  if (message.type === 'FETCH_SERVER_SETTINGS') {
    fetchServerSettings()
      .then(settings => sendResponse({ success: true, settings }))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  }

  // Проверка магазина перед импортом
  if (message.type === 'VALIDATE_SHOP') {
    validateShop(message.avito_user_id, message.shop_name)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  }

  // Импорт заказов
  if (message.type === 'IMPORT_ORDERS') {
    importOrders(message.orders)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  }

  if (message.type === 'FETCH_PAGE') {
    // SECURITY: Whitelist доменов для fetch
    const ALLOWED_FETCH_DOMAINS = ['www.avito.ru', 'avito.ru', 'm.avito.ru'];
    try {
      const urlObj = new URL(message.url);
      if (!ALLOWED_FETCH_DOMAINS.includes(urlObj.hostname)) {
        sendResponse({ success: false, error: 'Domain not allowed' });
        return true;
      }
      // Только HTTPS
      if (urlObj.protocol !== 'https:') {
        sendResponse({ success: false, error: 'Only HTTPS allowed' });
        return true;
      }
    } catch (e) {
      sendResponse({ success: false, error: 'Invalid URL' });
      return true;
    }

    fetch(message.url, {
      credentials: 'include',
      headers: message.headers || {}
    })
      .then(response => response.text())
      .then(html => sendResponse({ success: true, html }))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  }
});

// Загрузка настроек с сервера AcreOne
async function fetchServerSettings() {
  const { acreone_api_url, acreone_token } = await chrome.storage.local.get([
    'acreone_api_url',
    'acreone_token'
  ]);

  if (!acreone_token) {
    throw new Error('Токен не настроен');
  }

  const response = await fetch(`${acreone_api_url}/rest/orders/settings`, {
    headers: {
      'Authorization': `Bearer ${acreone_token}`
    }
  });

  if (!response.ok) {
    if (response.status === 401) throw new Error('Неверный токен');
    throw new Error(`Ошибка сервера: ${response.status}`);
  }

  const settings = await response.json();

  // Получаем текущее значение AI (хранится только локально)
  const { use_ai_analysis } = await chrome.storage.local.get(['use_ai_analysis']);

  // Сохраняем настройки локально (use_ai_analysis не перезаписываем с сервера)
  await chrome.storage.local.set({
    collect_size: settings.collect_size ?? true,
    collect_article: settings.collect_article ?? true,
    collect_color: settings.collect_color ?? true,
    collect_buyer_name: settings.collect_buyer_name ?? true,
    collect_photo: settings.collect_photo ?? true,
    allowed_shops: settings.allowed_shops || [],
    max_shops: settings.max_shops || 1
  });

  // Возвращаем настройки с локальным значением AI
  return { ...settings, use_ai_analysis: use_ai_analysis ?? false };
}

// Проверка магазина на сервере
async function validateShop(avitoUserId, shopName) {
  const { acreone_api_url, acreone_token } = await chrome.storage.local.get([
    'acreone_api_url',
    'acreone_token'
  ]);

  if (!acreone_token) {
    return { success: false, error: 'Токен не настроен' };
  }

  try {
    const response = await fetch(`${acreone_api_url}/rest/orders/validate-shop`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${acreone_token}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ avito_user_id: avitoUserId, shop_name: shopName })
    });

    const result = await response.json();

    if (!response.ok) {
      return { success: false, allowed: false, error: result.error || 'Ошибка проверки' };
    }

    // Обновляем локальные данные о магазинах
    if (result.newShopAdded) {
      await fetchServerSettings();
    }

    return { success: true, ...result };

  } catch (error) {
    console.error('[AcreOne] Ошибка проверки магазина:', error);
    return { success: false, error: error.message };
  }
}

// Импорт заказов на сервер
async function importOrders(orders) {
  const { acreone_api_url, acreone_token } = await chrome.storage.local.get([
    'acreone_api_url',
    'acreone_token'
  ]);

  // SECURITY: Не логируем детали токена

  if (!acreone_token) {
    return { success: false, error: 'Токен не настроен. Откройте настройки расширения.' };
  }

  try {
    const response = await fetch(`${acreone_api_url}/rest/orders/import`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${acreone_token}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ orders })
    });

    const result = await response.json();

    // Особая обработка лимита магазинов (402)
    if (response.status === 402) {
      return {
        success: false,
        shopLimitReached: true,
        error: result.message || 'Достигнут лимит магазинов',
        ...result
      };
    }

    if (!response.ok) {
      return { success: false, error: result.error || 'Ошибка импорта' };
    }

    // Обновляем статистику
    const stats = await chrome.storage.local.get(['stats_sent']);
    await chrome.storage.local.set({
      stats_sent: (stats.stats_sent || 0) + (result.imported || 0)
    });

    return { success: true, ...result };

  } catch (error) {
    console.error('[AcreOne BG] Ошибка импорта:', error);
    // Более понятные сообщения об ошибках
    let errorMsg = error.message;
    if (error.message.includes('Failed to fetch') || error.message.includes('NetworkError')) {
      errorMsg = 'Нет связи с сервером. Проверьте интернет.';
    } else if (error.message.includes('timeout')) {
      errorMsg = 'Сервер не отвечает. Попробуйте позже.';
    }
    return { success: false, error: errorMsg };
  }
}

// Обработка клика по контекстному меню
chrome.contextMenus.onClicked.addListener((info, tab) => {
  if (info.menuItemId === 'acreone-collect') {
    chrome.tabs.sendMessage(tab.id, { type: 'COLLECT_CURRENT_ORDER' });
  }
});
