/**
 * AcreOne - Сборщик заказов Авито
 * Content script для страницы списка заказов
 * v2.1.4 - Security hardening (XSS protection, escapeHtml)
 */

(function() {
  'use strict';

  // === ЗАЩИТА ===
  // Проверка домена (СТРОГАЯ)
  const ALLOWED_DOMAINS = ['www.avito.ru', 'avito.ru'];
  if (!ALLOWED_DOMAINS.includes(window.location.hostname)) {
    return; // Тихий выход без логов
  }

  // Проверка что это расширение Chrome/Yandex
  if (typeof chrome === 'undefined' || !chrome.runtime || !chrome.runtime.id) {
    return; // Тихий выход
  }

  // Проверка протокола (только HTTPS)
  if (window.location.protocol !== 'https:') {
    return;
  }

  // Защита от копирования в консоль
  const _log = console.log;
  const _error = console.error;

  // Проверка целостности (простая)
  const EXTENSION_SIGNATURE = 'acreone-v2.1.4-secure';
  if (window.__acreone_loaded === EXTENSION_SIGNATURE) {
    return; // Уже загружено
  }
  window.__acreone_loaded = EXTENSION_SIGNATURE;

  // === SECURITY MODULE v2.1 ===
  const Security = window.__AcreOneSecurity;
  if (!Security || Security.VERSION !== '2.1') {
    console.error('[AcreOne] Security module not loaded or version mismatch');
    return;
  }

  // Защита от внешних скриптов - не даём читать наши данные
  Object.defineProperty(window, '__acreone_data', {
    value: undefined,
    writable: false,
    configurable: false
  });

  // === SECURITY: HTML escaping для защиты от XSS ===
  function escapeHtml(text) {
    if (text === null || text === undefined) return '';
    const str = String(text);
    const map = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#039;'
    };
    return str.replace(/[&<>"']/g, c => map[c]);
  }

  // === СОСТОЯНИЕ ===
  // Массив собранных заказов (защита на уровне подписи при отправке)
  let collectedOrders = [];
  let isCollecting = false;
  let collectSettings = {};  // Настройки сбора с сервера
  let currentShopId = null;  // ID текущего магазина (avito_user_id)
  let currentShopName = null;
  let isAuthorized = false;  // Флаг авторизации
  let isExtensionEnabled = true; // Флаг включения расширения
  let useAiAnalysis = false; // Флаг использования AI для анализа

  // === КОНФИГУРАЦИЯ ТРАНСПОРТНЫХ КОМПАНИЙ ===
  // Унифицированные правила для парсинга треков
  const DELIVERY_COMPANIES = {
    'avito': {
      names: ['Авито', 'Avito', 'авито', 'avito-pvz', 'avito_delivery', 'пункт Авито', 'Авито Доставка'],
      trackPatterns: [
        /sellerDispatchNumber[^}]*"original":\s*"(\d{10,15})"/i,
        /dispatchNumber['":\s]+['"]?(\d{10,15})['"]?/i
      ],
      trackLength: { min: 10, max: 15 },
      trackPrefix: null
    },
    'yandex': {
      names: ['Яндекс', 'Яндекс Доставка', 'Yandex', 'yandex'],
      trackPatterns: [
        // P + 9-12 цифр (гибкий формат)
        /["']?(P\d{9,12})["']?/i,
        /yandex[^}]*trackingNumber['":\s]+['"]?(P?\d{9,15})['"]?/i
      ],
      trackLength: { min: 10, max: 14 },
      trackPrefix: 'P'
    },
    '5post': {
      names: ['5Post', '5post', '5 Post', '5 post', '5POST'],
      trackPatterns: [
        // Код отправки 5Post для QR (9 цифр: 093 744 307) - это главный код для постаматов
        /dispatchNumber['":\s]+['"]?(\d{3}\s*\d{3}\s*\d{3})['"]?/i,
        /sellerDispatchNumber[^}]*"original":\s*"(\d{3}\s*\d{3}\s*\d{3})"/i,
        // 9-значный код с пробелами или без (формат 093 744 307 или 093744307)
        /(\d{3})\s+(\d{3})\s+(\d{3})/,
        // Fallback: любой 9-значный номер
        /["']?(\d{9})["']?/
      ],
      trackLength: { min: 9, max: 9 },
      useQrCode: true // 5Post использует QR-код вместо штрих-кода
    },
    'cdek': {
      names: ['СДЭК', 'CDEK', 'сдэк', 'cdek'],
      trackPatterns: [
        /trackingNumber['":\s]+['"]?(\d{10,12})['"]?/i,
        /cdek[^}]*number['":\s]+['"]?(\d{10,12})['"]?/i
      ],
      trackLength: { min: 10, max: 12 },
      trackPrefix: null
    },
    'pochta': {
      names: ['Почта России', 'Почта', 'pochta', 'russian post'],
      trackPatterns: [
        /trackingNumber['":\s]+['"]?(\d{14})['"]?/i,
        /barcode['":\s]+['"]?(\d{14})['"]?/i
      ],
      trackLength: { min: 14, max: 14 },
      trackPrefix: null
    },
    'boxberry': {
      names: ['Boxberry', 'boxberry', 'Боксберри'],
      trackPatterns: [
        /trackingNumber['":\s]+['"]?([A-Z0-9]{10,15})['"]?/i
      ],
      trackLength: { min: 10, max: 15 },
      trackPrefix: null
    },
    'dpd': {
      names: ['DPD', 'dpd', 'ДПД'],
      trackPatterns: [
        // DPD использует sellerDispatchNumber.original как Авито Доставка
        /sellerDispatchNumber[^}]*"original":\s*"(RU\d{9})"/i,
        /dispatchNumber['":\s]+['"]?(RU\d{9})['"]?/i,
        // Общий паттерн RU + 9 цифр
        /"(RU\d{9})"/i,
        /\b(RU\d{9})\b/i
      ],
      trackLength: { min: 11, max: 11 },
      trackPrefix: 'RU'
    },
    'pek': {
      names: ['ПЭК', 'PEK', 'пэк'],
      trackPatterns: [
        /trackingNumber['":\s]+['"]?(\d{8,15})['"]?/i
      ],
      trackLength: { min: 8, max: 15 },
      trackPrefix: null
    },
    'dostavista': {
      names: ['Dostavista', 'Достависта', 'dostavista', 'достависта', 'Курьер'],
      trackPatterns: [
        // Номер Достависты: 142 495 335 (9 цифр с пробелами или без)
        /Номер Достависты[:\s]*([0-9\s]{9,15})/i,
        /dostavista[^}]*number['":\s]+['"]?(\d{9,12})['"]?/i,
        // 9 цифр подряд или с пробелами
        /(\d{3}\s*\d{3}\s*\d{3})/
      ],
      trackLength: { min: 9, max: 12 },
      trackPrefix: null,
      isCourier: true // Курьерская доставка
    }
  };

  // Универсальные паттерны для fallback
  const UNIVERSAL_TRACK_PATTERNS = [
    /trackingNumber['":\s]+['"]?([A-Z0-9]{6,20})['"]?/i,
    /tracking['":\s]+['"]?([A-Z0-9]{10,20})['"]?/i,
    /shipmentNumber['":\s]+['"]?([A-Z0-9]{6,20})['"]?/i,
    /barcode['":\s]+['"]?(\d{10,20})['"]?/i,
    /waybillNumber['":\s]+['"]?([A-Z0-9]{6,20})['"]?/i,
    /"number":\s*"(\d{10,15})"/,
    /"code":\s*"(\d{10,15})"/
  ];

  // === ПРОВЕРКА TOGGLE ===
  async function checkExtensionEnabled() {
    try {
      const settings = await chrome.storage.local.get(['extension_enabled']);
      isExtensionEnabled = settings.extension_enabled !== false; // По умолчанию включено
      return isExtensionEnabled;
    } catch (e) {
      return true; // По умолчанию включено
    }
  }

  // Слушаем изменения toggle из popup
  chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.type === 'TOGGLE_EXTENSION') {
      isExtensionEnabled = request.enabled;
      if (isExtensionEnabled) {
        // Инициализируем UI при включении
        init();
      } else {
        // Скрываем UI при выключении
        hideUI();
      }
      sendResponse({ success: true });
    }
    return true;
  });

  function hideUI() {
    const toolbar = document.getElementById('acreone-toolbar');
    if (toolbar) toolbar.style.display = 'none';
    document.body.classList.remove('acreone-toolbar-active');

    const panel = document.getElementById('acreone-panel');
    if (panel) panel.style.display = 'none';
  }

  function showUI() {
    const toolbar = document.getElementById('acreone-toolbar');
    if (toolbar) {
      toolbar.style.display = '';
      document.body.classList.add('acreone-toolbar-active');
      updateToolbarInfo(); // Обновляем инфо при показе
    }

    const panel = document.getElementById('acreone-panel');
    if (panel) panel.style.display = '';
  }

  // === ПРОВЕРКА КОНТЕКСТА РАСШИРЕНИЯ ===
  function isExtensionContextValid() {
    try {
      // Проверяем что runtime всё ещё доступен
      return !!(chrome && chrome.runtime && chrome.runtime.id);
    } catch (e) {
      return false;
    }
  }

  // Показ сообщения о необходимости обновить страницу
  function showExtensionInvalidError() {
    // Удаляем старые элементы AcreOne
    const oldToolbar = document.getElementById('acreone-toolbar');
    if (oldToolbar) oldToolbar.remove();
    document.body.classList.remove('acreone-toolbar-active');

    const oldPanel = document.getElementById('acreone-panel');
    if (oldPanel) oldPanel.remove();

    const oldStatus = document.querySelector('.acreone-status');
    if (oldStatus) oldStatus.remove();

    // Создаём уведомление о перезагрузке
    const notification = document.createElement('div');
    notification.className = 'acreone-reload-notification';
    notification.innerHTML = `
      <div class="acreone-reload-content">
        <div class="acreone-reload-icon">
          <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="#f97316" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <polyline points="23 4 23 10 17 10"></polyline>
            <polyline points="1 20 1 14 7 14"></polyline>
            <path d="M3.51 9a9 9 0 0 1 14.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0 0 20.49 15"></path>
          </svg>
        </div>
        <div class="acreone-reload-text">
          <strong>AcreOne обновлён</strong>
          <p>Обновите страницу для продолжения работы</p>
        </div>
        <button class="acreone-reload-btn" onclick="location.reload()">
          Обновить
        </button>
      </div>
    `;
    document.body.appendChild(notification);

    // Добавляем стили для уведомления
    const style = document.createElement('style');
    style.textContent = `
      .acreone-reload-notification {
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 999999;
        background: rgba(17, 24, 39, 0.98);
        border: 1px solid rgba(249, 115, 22, 0.5);
        border-radius: 12px;
        padding: 16px 20px;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.4);
        backdrop-filter: blur(20px);
        animation: acreone-slide-in 0.3s ease;
      }
      @keyframes acreone-slide-in {
        from { transform: translateX(100px); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
      }
      .acreone-reload-content {
        display: flex;
        align-items: center;
        gap: 12px;
      }
      .acreone-reload-icon {
        display: flex;
        align-items: center;
        justify-content: center;
      }
      .acreone-reload-icon svg {
        animation: acreone-spin 2s linear infinite;
      }
      .acreone-reload-text {
        color: #e5e7eb;
      }
      .acreone-reload-text strong {
        display: block;
        font-size: 14px;
        color: #fff;
      }
      .acreone-reload-text p {
        font-size: 12px;
        color: #9ca3af;
        margin: 4px 0 0 0;
      }
      .acreone-reload-btn {
        background: linear-gradient(135deg, #f97316 0%, #ea580c 100%);
        color: white;
        border: none;
        padding: 8px 16px;
        border-radius: 8px;
        font-weight: 600;
        font-size: 13px;
        cursor: pointer;
        transition: transform 0.2s;
      }
      .acreone-reload-btn:hover {
        transform: scale(1.05);
      }
    `;
    document.head.appendChild(style);
  }

  // Показ сообщения что магазин не разрешён
  let shopNotAllowedShown = false;
  function showShopNotAllowedError(reason, currentShops) {
    if (shopNotAllowedShown) return;
    shopNotAllowedShown = true;

    // Удаляем старые элементы AcreOne
    const oldToolbar = document.getElementById('acreone-toolbar');
    if (oldToolbar) oldToolbar.remove();
    document.body.classList.remove('acreone-toolbar-active');

    const oldPanel = document.getElementById('acreone-panel');
    if (oldPanel) oldPanel.remove();


    // Создаём уведомление
    const notification = document.createElement('div');
    notification.id = 'acreone-shop-blocked';
    notification.className = 'acreone-shop-blocked-notification';

    // SECURITY: Экранируем данные от сервера
    const safeReason = escapeHtml(reason || 'Достигнут лимит магазинов');
    const shopsList = currentShops?.length > 0
      ? currentShops.map(s => `• ${escapeHtml(s.name || s.id)}`).join('<br>')
      : '';

    notification.innerHTML = `
      <div class="acreone-blocked-content">
        <div class="acreone-blocked-icon">
          <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="#ef4444" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <circle cx="12" cy="12" r="10"></circle>
            <line x1="4.93" y1="4.93" x2="19.07" y2="19.07"></line>
          </svg>
        </div>
        <div class="acreone-blocked-text">
          <strong>Магазин недоступен</strong>
          <p>${safeReason}</p>
          ${shopsList ? `<div class="acreone-shops-list">Разрешённые магазины:<br>${shopsList}</div>` : ''}
        </div>
        <button class="acreone-blocked-close" onclick="this.closest('.acreone-shop-blocked-notification').remove()">
          ✕
        </button>
      </div>
    `;
    document.body.appendChild(notification);

    // Добавляем стили
    if (!document.getElementById('acreone-blocked-styles')) {
      const style = document.createElement('style');
      style.id = 'acreone-blocked-styles';
      style.textContent = `
        .acreone-shop-blocked-notification {
          position: fixed;
          top: 20px;
          right: 20px;
          z-index: 999999;
          background: rgba(17, 24, 39, 0.98);
          border: 1px solid rgba(239, 68, 68, 0.5);
          border-radius: 12px;
          padding: 16px 20px;
          box-shadow: 0 8px 32px rgba(0, 0, 0, 0.4);
          backdrop-filter: blur(20px);
          animation: acreone-slide-in 0.3s ease;
          max-width: 350px;
        }
        .acreone-blocked-content {
          display: flex;
          align-items: flex-start;
          gap: 12px;
        }
        .acreone-blocked-icon {
          display: flex;
          align-items: center;
          justify-content: center;
          flex-shrink: 0;
          margin-top: 2px;
        }
        .acreone-blocked-text {
          color: #e5e7eb;
          flex: 1;
        }
        .acreone-blocked-text strong {
          display: block;
          font-size: 14px;
          color: #fff;
        }
        .acreone-blocked-text p {
          font-size: 12px;
          color: #9ca3af;
          margin: 4px 0 0 0;
        }
        .acreone-shops-list {
          font-size: 11px;
          color: #6b7280;
          margin-top: 8px;
          padding-top: 8px;
          border-top: 1px solid rgba(255,255,255,0.1);
        }
        .acreone-blocked-close {
          background: none;
          border: none;
          color: #6b7280;
          font-size: 16px;
          cursor: pointer;
          padding: 0;
          line-height: 1;
          flex-shrink: 0;
        }
        .acreone-blocked-close:hover {
          color: #9ca3af;
        }
      `;
      document.head.appendChild(style);
    }
  }

  // Проверка магазина через API
  async function validateShopWithServer(shopId, shopName) {
    if (!shopId) {
      return { allowed: false, reason: 'Shop ID не определён' };
    }

    // Проверяем контекст расширения
    if (!isExtensionContextValid()) {
      return { allowed: false, reason: 'Контекст расширения недействителен' };
    }

    try {
      // Проверяем есть ли токен
      const { acreone_token } = await chrome.storage.local.get(['acreone_token']);
      if (!acreone_token) {
        // Если токен не настроен - не показываем UI вообще
        return { allowed: false, reason: 'Токен не настроен. Откройте настройки расширения.' };
      }

      // Отправляем запрос через background script с таймаутом
      const response = await Promise.race([
        chrome.runtime.sendMessage({
          type: 'VALIDATE_SHOP',
          avito_user_id: shopId,
          shop_name: shopName
        }),
        new Promise((_, reject) => setTimeout(() => reject(new Error('Timeout')), 15000))
      ]);


      // SECURITY v2.0: Offline mode ОТКЛЮЧЁН - требуем валидацию сервером
      if (!response) {
        return { allowed: false, reason: 'Сервер недоступен. Проверьте интернет-соединение.' };
      }

      if (response.success && response.allowed) {
        return { allowed: true, newShopAdded: response.newShopAdded };
      }

      return {
        allowed: false,
        reason: response.reason || response.error || 'Магазин не разрешён',
        currentShops: response.currentShops
      };

    } catch (error) {
      console.error('[AcreOne] Ошибка проверки магазина:', error);
      // SECURITY v2.0: Offline mode ОТКЛЮЧЁН - требуем валидацию сервером
      return { allowed: false, reason: 'Ошибка связи с сервером: ' + error.message };
    }
  }

  // === ИНИЦИАЛИЗАЦИЯ ===
  async function init() {
    // Проверяем включено ли расширение
    const enabled = await checkExtensionEnabled();
    if (!enabled) {
      hideUI();
      return;
    }


    // Загружаем настройки и ID магазина
    await loadSettings();
    await extractCurrentShopInfo();

    // Магазин НЕ валидируем при загрузке - только при нажатии "Собрать заказы"
    // Это позволяет пользователю видеть UI даже если лимит магазинов исчерпан

    // Показываем UI сразу
    showUI();

    // Ждём загрузки страницы (SPA - React грузится асинхронно)
    waitForOrders();
  }

  // Загрузка настроек из storage
  async function loadSettings() {
    // Проверяем контекст расширения
    if (!isExtensionContextValid()) {
      showExtensionInvalidError();
      return;
    }

    try {
      const settings = await chrome.storage.local.get([
        'acreone_token',
        'acreone_api_url',
        'collect_size',
        'collect_article',
        'collect_color',
        'collect_buyer_name',
        'collect_photo',
        'use_ai_analysis'
      ]);

      // Проверяем авторизацию
      isAuthorized = !!settings.acreone_token;
      useAiAnalysis = settings.use_ai_analysis === true;

      collectSettings = settings;
    } catch (e) {
      // Проверяем тип ошибки
      if (e.message && e.message.includes('Extension context invalidated')) {
        showExtensionInvalidError();
        return;
      }
      console.error('[AcreOne] Ошибка загрузки настроек:', e);
    }
  }

  // Извлечение ID и названия текущего магазина
  async function extractCurrentShopInfo() {
    try {
      // Метод 1 (ПРИОРИТЕТ): Из JWT токена sessid в cookies
      // sessid содержит payload с полем "u" = userId
      if (!currentShopId) {
        const sessidMatch = document.cookie.match(/sessid=([^;]+)/);
        if (sessidMatch) {
          try {
            const parts = sessidMatch[1].split('.');
            if (parts.length >= 2) {
              // Декодируем base64 payload JWT (поддержка URL-safe base64)
              let base64 = parts[1];
              // URL-safe base64: заменяем - на + и _ на /
              base64 = base64.replace(/-/g, '+').replace(/_/g, '/');
              // Добавляем padding если нужно
              while (base64.length % 4) {
                base64 += '=';
              }
              const payload = JSON.parse(atob(base64));
              const userId = payload.u || payload.user_id || payload.userId || payload.id || payload.sub;
              if (userId && userId > 1000000) {
                currentShopId = userId;
              }
            }
          } catch (e) {
            // Silent fail
          }
        }
      }

      // Метод 1.5: Из других cookies (buyer, user)
      if (!currentShopId) {
        const cookiesToCheck = ['buyer', 'user', 'u', 'uid'];
        for (const cookieName of cookiesToCheck) {
          const cookieMatch = document.cookie.match(new RegExp(`${cookieName}=([^;]+)`));
          if (cookieMatch) {
            const value = cookieMatch[1];
            // Если это число - используем напрямую
            if (/^\d{7,12}$/.test(value)) {
              currentShopId = parseInt(value);
              break;
            }
          }
        }
      }

      // Метод 2: Из __NEXT_DATA__ (React)
      if (!currentShopId) {
        const nextDataEl = document.getElementById('__NEXT_DATA__');
        if (nextDataEl) {
          const data = JSON.parse(nextDataEl.textContent);
          const userId = data?.props?.pageProps?.user?.id ||
                         data?.props?.initialState?.user?.id ||
                         data?.props?.pageProps?.userId;
          if (userId) {
            currentShopId = userId;
          }

          // Также пробуем получить название
          const userName = data?.props?.pageProps?.user?.name ||
                          data?.props?.initialState?.user?.name ||
                          data?.props?.pageProps?.profileName;
          if (userName) {
            currentShopName = userName;
          }
        }
      }

      // Метод 3: Получаем ID и название магазина через API profileinfo (ГЛАВНЫЙ МЕТОД)
      if (!currentShopId || !currentShopName) {
        try {
          const response = await fetch('https://www.avito.ru/web/2/profileinfo', {
            method: 'POST',
            credentials: 'include',
            headers: { 'Content-Type': 'application/json' },
            body: '{}'
          });
          if (response.ok) {
            const data = await response.json();

            // Извлекаем ID из разных возможных полей
            const profileId = data.profile?.id || data.profile?.userId || data.id || data.userId || data.user_id;
            if (profileId && !currentShopId) {
              currentShopId = parseInt(profileId);
            }

            // Извлекаем название
            if (data.profile?.name && !currentShopName) {
              currentShopName = data.profile.name;
            }
          }
        } catch (e) {
          // Silent fail
        }
      }

      // Метод 3.5: Загружаем страницу profile/extended и парсим ID оттуда (НАДЁЖНЫЙ)
      if (!currentShopId) {
        try {
          const response = await fetch('https://www.avito.ru/profile/extended', {
            method: 'GET',
            credentials: 'include'
          });
          if (response.ok) {
            const html = await response.text();

            // ГЛАВНЫЙ МЕТОД: Ищем ID в URL-encoded JSON
            const urlEncodedMatch = html.match(/%22id%22%3A(\d{7,12})%2C%22isAdvertisingAgency%22/);
            if (urlEncodedMatch) {
              currentShopId = parseInt(urlEncodedMatch[1]);
            } else {
              // Fallback: ищем по контексту hashedId
              const hashedIdMatch = html.match(/%22hashedId%22%3A%22[a-f0-9]+%22%2C%22id%22%3A(\d+)%2C/);
              if (hashedIdMatch) {
                currentShopId = parseInt(hashedIdMatch[1]);
              } else {
                // Последний fallback: текстовый поиск "ID 104 475 286"
                const textContent = html.replace(/<[^>]+>/g, ' ').replace(/\s+/g, ' ');
                const textMatch = textContent.match(/ID\s+(\d{1,3}\s+\d{3}\s+\d{3})/i);
                if (textMatch) {
                  const idStr = textMatch[1].replace(/\s/g, '');
                  currentShopId = parseInt(idStr);
                }
              }
            }
          }
        } catch (e) {
          // Silent fail
        }
      }

      // Метод 5: Из ссылки на профиль (fallback)
      if (!currentShopId) {
        const profileLink = document.querySelector('a[href*="/profile/"], a[href*="/user/"]');
        if (profileLink) {
          const match = profileLink.href.match(/\/(profile|user)\/(\d+)/);
          if (match) {
            currentShopId = parseInt(match[2]);
          }
        }
      }

      // Метод 6: Из window.__INITIAL_STATE__ (часто есть на Avito)
      if (!currentShopId) {
        try {
          const scripts = document.querySelectorAll('script:not([src])');
          for (const script of scripts) {
            const content = script.textContent || '';
            const userIdMatch = content.match(/["']?userId["']?\s*[:=]\s*(\d{7,12})/);
            if (userIdMatch) {
              currentShopId = parseInt(userIdMatch[1]);
              break;
            }
            const altMatch = content.match(/["']?user_id["']?\s*[:=]\s*(\d{7,12})/);
            if (altMatch) {
              currentShopId = parseInt(altMatch[1]);
              break;
            }
          }
        } catch (e) {
          // Silent fail
        }
      }

      // Метод 7: Извлекаем название магазина из DOM (fallback)
      if (!currentShopName) {
        const shopNameEl = document.querySelector(
          '[class*="ProfileName"], [class*="profile-name"], [data-marker="profile-name"], ' +
          '[class*="HeaderUserMenu"] span, [data-marker="header/user-avatar"]'
        );
        if (shopNameEl) {
          const text = shopNameEl.textContent?.trim();
          if (text && text.length > 1 && text.length < 100) {
            currentShopName = text;
          }
        }
      }

    } catch (e) {
      console.error('[AcreOne] Ошибка извлечения данных магазина:', e);
    }
  }

  // Сохранение информации о магазине в storage
  async function saveShopInfo() {
    try {
      // Получаем shop_id и shop_name из собранных заказов или глобальных переменных
      let shopId = currentShopId;
      let shopName = currentShopName;

      // Если не нашли глобально - берём из первого заказа
      if (!shopId && collectedOrders.length > 0) {
        shopId = collectedOrders[0].shop_id || collectedOrders[0].avito_user_id;
        shopName = collectedOrders[0].shop_name || collectedOrders[0].avito_shop_name;
      }

      if (!shopId) {
        return;
      }

      // Загружаем текущий список магазинов
      const { allowed_shops = [], max_shops = 1 } = await chrome.storage.local.get(['allowed_shops', 'max_shops']);

      // Проверяем, есть ли уже этот магазин (приводим к строке для корректного сравнения)
      const shopIdStr = String(shopId);
      const existingIndex = allowed_shops.findIndex(s => String(s.avito_user_id) === shopIdStr);

      if (existingIndex >= 0) {
        // Обновляем название если изменилось
        if (shopName && allowed_shops[existingIndex].avito_shop_name !== shopName) {
          allowed_shops[existingIndex].avito_shop_name = shopName;
          await chrome.storage.local.set({ allowed_shops });
        }
      } else {
        // Добавляем новый магазин
        allowed_shops.push({
          avito_user_id: shopId,
          avito_shop_name: shopName || `Магазин ${shopId}`
        });
        await chrome.storage.local.set({ allowed_shops });
      }

      // Обновляем глобальные переменные
      currentShopId = shopId;
      if (shopName) currentShopName = shopName;

    } catch (e) {
      console.error('[AcreOne] Ошибка сохранения магазина:', e);
    }
  }

  // Ждём пока React отрендерит заказы
  function waitForOrders() {
    let attempts = 0;
    const maxAttempts = 30; // 15 секунд максимум

    const checkInterval = setInterval(() => {
      attempts++;

      // Ищем любые признаки загруженных заказов
      const hasOrders = document.body.textContent.includes('Отправьте заказ') ||
                        document.body.textContent.includes('Едет к покупателю') ||
                        document.body.textContent.includes('Выдан покупателю') ||
                        document.querySelectorAll('[data-marker]').length > 5;

      if (hasOrders || attempts >= maxAttempts) {
        clearInterval(checkInterval);

        onReady();
      }
    }, 500);
  }

  function onReady() {
    // Добавляем кнопку сбора
    createCollectButton();

    // Создаём боковую панель
    createSidePanel();

    // Подсвечиваем заказы "Отправьте заказ"
    setTimeout(() => {
      highlightShippableOrders();
    }, 1000);

  }

  // Создание кнопки сбора заказов
  function createCollectButton() {
    // Ждём появления кнопки "Служба доставки" (React загружает её асинхронно)
    let attempts = 0;
    const maxAttempts = 20; // 10 секунд максимум

    const tryCreateButton = () => {
      attempts++;

      // Ищем кнопку "Служба доставки"
      let deliveryButton = null;
      const allButtons = document.querySelectorAll('button[class*="styles-module-root"]');

      for (const b of allButtons) {
        if (b.textContent.trim() === 'Служба доставки') {
          deliveryButton = b;
          break;
        }
      }

      // Если кнопка не найдена и есть попытки - ждём
      if (!deliveryButton && attempts < maxAttempts) {
        setTimeout(tryCreateButton, 500);
        return;
      }

      // Проверяем что наша кнопка ещё не создана
      if (document.getElementById('acreone-collect-btn')) {
        return;
      }

      // Создаём кнопку в стиле Авито
      const btn = document.createElement('button');
      btn.id = 'acreone-collect-btn';

      // SVG иконка коробки
      const boxIcon = `<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"></path>
        <polyline points="3.27 6.96 12 12.01 20.73 6.96"></polyline>
        <line x1="12" y1="22.08" x2="12" y2="12"></line>
      </svg>`;

      btn.innerHTML = `${boxIcon}<span>Собрать заказы</span>`;
      btn.onclick = startCollection;

      if (deliveryButton) {
        // Копируем классы от кнопки Авито для идентичного стиля
        btn.className = deliveryButton.className + ' acreone-collect-btn';

        // Структура: дедушка (flex row) > родитель (DIV block) > кнопка
        // Родитель имеет display:block, поэтому нужно создать свой DIV-обёртку
        // и добавить в дедушку как 4-й элемент в ряд
        const parent = deliveryButton.parentElement;
        const grandpa = parent.parentElement;

        // Создаём обёртку с теми же классами что у родителя
        const wrapper = document.createElement('div');
        wrapper.className = parent.className;
        wrapper.appendChild(btn);

        // Добавляем в дедушку (flex контейнер)
        grandpa.appendChild(wrapper);
      }
    };

    // Запускаем поиск кнопки
    tryCreateButton();
  }

  // Получить инфо о доставке из заказов
  function getDeliveryInfo(orders) {
    const companies = {};

    orders.forEach(order => {
      const text = order.element.textContent || '';
      let company = detectDeliveryCompany(text);
      if (company) {
        companies[company] = (companies[company] || 0) + 1;
      }
    });

    // Возвращаем самую частую ТК
    let maxCount = 0;
    let mainCompany = null;
    for (const [name, count] of Object.entries(companies)) {
      if (count > maxCount) {
        maxCount = count;
        mainCompany = name;
      }
    }

    return {
      name: mainCompany,
      count: maxCount,
      all: companies
    };
  }

  // Обновить инфо в тулбаре
  function updateToolbarInfo() {
    const toolbar = document.getElementById('acreone-toolbar');
    if (!toolbar) return;

    const orders = findShippableOrders();
    const deliveryInfo = getDeliveryInfo(orders);

    const infoEl = toolbar.querySelector('.acreone-delivery-info');
    if (infoEl) {
      infoEl.innerHTML = `
        <span class="delivery-icon">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <rect x="1" y="3" width="15" height="13"></rect>
            <polygon points="16 8 20 8 23 11 23 16 16 16 16 8"></polygon>
            <circle cx="5.5" cy="18.5" r="2.5"></circle>
            <circle cx="18.5" cy="18.5" r="2.5"></circle>
          </svg>
        </span>
        <span class="delivery-name">${deliveryInfo.name || 'Не определено'}</span>
        <span class="delivery-count">${orders.length} заказов</span>
      `;
    }
  }

  // Создание боковой панели
  function createSidePanel() {
    const panel = document.createElement('div');
    panel.className = 'acreone-panel';
    panel.id = 'acreone-panel';

    // Инлайн-стили для гарантированного применения (Avito может перебивать CSS)
    panel.style.cssText = `
      position: fixed !important;
      top: 12px !important;
      right: 12px !important;
      width: 420px !important;
      height: calc(100vh - 24px) !important;
      background: rgba(10, 10, 15, 0.98) !important;
      border: 1px solid rgba(255, 255, 255, 0.1) !important;
      border-radius: 20px !important;
      box-shadow: -8px 0 40px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(255, 255, 255, 0.05) !important;
      z-index: 10001 !important;
      transform: translateX(calc(100% + 24px));
      transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1) !important;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif !important;
      backdrop-filter: blur(40px) !important;
      -webkit-backdrop-filter: blur(40px) !important;
      overflow: hidden !important;
    `;

    panel.innerHTML = `
      <div class="acreone-panel-bg"></div>
      <div class="acreone-panel-header">
        <h3>Собранные заказы</h3>
        <button class="acreone-panel-close" onclick="document.getElementById('acreone-panel').classList.remove('open')">&times;</button>
      </div>
      <div class="acreone-panel-content" id="acreone-orders-list">
        <p style="color: #666; text-align: center;">Нажмите "Собрать заказы"</p>
      </div>
      <div class="acreone-panel-footer">
        <button class="acreone-send-btn" id="acreone-send-btn" disabled>
          Отправить в AcreOne (0)
        </button>
      </div>
    `;

    document.body.appendChild(panel);

    // Добавляем анимированный фон отдельным элементом
    const bgEl = panel.querySelector('.acreone-panel-bg');
    if (bgEl) {
      bgEl.style.cssText = `
        position: absolute !important;
        top: 0 !important;
        left: 0 !important;
        right: 0 !important;
        bottom: 0 !important;
        background:
          radial-gradient(ellipse at 20% 20%, rgba(249, 115, 22, 0.15) 0%, transparent 50%),
          radial-gradient(ellipse at 80% 80%, rgba(234, 88, 12, 0.1) 0%, transparent 50%),
          radial-gradient(ellipse at 50% 50%, rgba(139, 92, 246, 0.05) 0%, transparent 60%) !important;
        animation: acreone-pulse-gradient 8s ease-in-out infinite !important;
        pointer-events: none !important;
        z-index: 0 !important;
        border-radius: 20px !important;
      `;
    }
  }

  // Подсветка заказов со статусом "Отправьте заказ"
  function highlightShippableOrders() {
    const orders = findShippableOrders();

    orders.forEach(order => {
      order.element.classList.add('acreone-order-highlight');
    });
  }

  // Поиск заказов со статусом "Отправьте заказ"
  function findShippableOrders() {
    const orders = [];

    // Ищем карточки заказов по data-marker="order-row"
    const orderCards = document.querySelectorAll('[data-marker="order-row"]');

    orderCards.forEach((card, idx) => {
      // Проверяем статус
      const statusEl = card.querySelector('[data-marker="order-status"]');
      const statusText = statusEl?.textContent?.trim() || '';
      const cardText = card.textContent || '';


      // Статусы для отправки (может быть разным для Авито Доставки)
      const shippingStatuses = [
        'Отправьте заказ',
        'Привезите на пункт',  // Авито Доставка
        'Доставьте заказ',
        'Готов к отправке',
        'Передайте курьеру'    // Dostavista / курьерская доставка
      ];

      if (shippingStatuses.some(s => statusText.includes(s))) {
        orders.push({
          element: card,
          status: 'shipping_required'
        });
      }
    });

    return orders;
  }

  // Начало сбора данных
  async function startCollection() {
    if (isCollecting) return;

    // Проверяем контекст расширения
    if (!isExtensionContextValid()) {
      showExtensionInvalidError();
      return;
    }

    const btn = document.getElementById('acreone-collect-btn');
    const boxIcon = `<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
      <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"></path>
      <polyline points="3.27 6.96 12 12.01 20.73 6.96"></polyline>
      <line x1="12" y1="22.08" x2="12" y2="12"></line>
    </svg>`;
    // Плавный спиннер без рывков
    const spinnerIcon = `<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" class="acreone-spin">
      <circle cx="12" cy="12" r="9" stroke-opacity="0.25"></circle>
      <path d="M12 3a9 9 0 0 1 9 9" stroke-opacity="1"></path>
    </svg>`;

    // Функция обновления текста кнопки с прогрессом
    const updateBtn = (text, isLoading = true, current = 0, total = 0) => {
      if (btn) {
        if (isLoading && total > 0) {
          // Показываем прогресс: спиннер + "Сбор 3/12"
          btn.innerHTML = `${spinnerIcon}<span class="acreone-progress">Сбор ${current}/${total}</span>`;
        } else {
          btn.innerHTML = `${isLoading ? spinnerIcon : boxIcon}<span>${text}</span>`;
        }
      }
    };

    // Проверяем авторизацию
    if (!isAuthorized) {
      updateBtn('Нет токена!', false);
      setTimeout(() => {
        btn.innerHTML = `${boxIcon}<span>Собрать заказы</span>`;
      }, 2000);
      return;
    }

    // Проверяем магазин на сервере ПЕРЕД сбором
    updateBtn('Проверка магазина...', true);

    const shopValidation = await validateShopWithServer(currentShopId, currentShopName);

    // SECURITY v2.0: Всегда требуем валидацию (offline mode отключён)
    if (!shopValidation.allowed) {
      // Магазин не разрешён - показываем ошибку
      updateBtn('Ошибка!', false);
      showShopNotAllowedError(shopValidation.reason, shopValidation.currentShops);
      setTimeout(() => {
        btn.innerHTML = `${boxIcon}<span>Собрать заказы</span>`;
      }, 2000);
      return;
    }

    isCollecting = true;
    collectedOrders = [];

    if (btn) {
      btn.classList.add('loading');
    }

    updateBtn('Поиск заказов...');

    try {
      const orders = findShippableOrders();

      if (orders.length === 0) {
        updateBtn('Нет заказов', false);
        setTimeout(() => {
          btn.innerHTML = `${boxIcon}<span>Собрать заказы</span>`;
          btn.classList.remove('loading');
        }, 2000);
        isCollecting = false;
        return;
      }

      // Начинаем сбор - показываем прогресс с 0
      updateBtn('', true, 0, orders.length);

      // Собираем данные с каждого заказа
      for (let i = 0; i < orders.length; i++) {
        updateBtn('', true, i + 1, orders.length);

        const orderItems = await parseOrderCard(orders[i].element);
        if (orderItems && Array.isArray(orderItems)) {
          // parseOrderCard возвращает МАССИВ товаров
          for (const item of orderItems) {
            collectedOrders.push(item);
          }
        }

        // Пауза между заказами (несколько запросов на заказ)
        await sleep(800);
      }

      const uniqueOrders = new Set(collectedOrders.map(o => o.order_number || o.avito_order_id)).size;
      updateBtn(`Готово! ${collectedOrders.length} шт.`, false);

      // Сохраняем информацию о магазине
      await saveShopInfo();

      // Показываем панель с результатами
      renderOrdersPanel();
      document.getElementById('acreone-panel').classList.add('open');

      // Возвращаем исходный текст через 2 сек
      setTimeout(() => {
        btn.innerHTML = `${boxIcon}<span>Собрать заказы</span>`;
      }, 2000);

    } catch (error) {
      console.error('[AcreOne] Ошибка сбора:', error);
      updateBtn('Ошибка!', false);
      setTimeout(() => {
        btn.innerHTML = `${boxIcon}<span>Собрать заказы</span>`;
      }, 2000);
    } finally {
      isCollecting = false;
      if (btn) {
        btn.classList.remove('loading');
      }
    }
  }

  // Парсинг карточки заказа
  // ВАЖНО: возвращает МАССИВ товаров (один заказ может содержать несколько товаров)
  async function parseOrderCard(cardElement) {
    try {
      const baseData = {
        // Базовые данные со страницы списка
        avito_order_id: null,
        order_number: null,  // Номер заказа (№70000000362055374)
        order_date: null,    // Дата заказа
        delivery_company: null,
        track_number: null,
        buyer_name: null,
        order_url: null,
        chat_url: null,
        // Данные магазина для ограничения
        avito_user_id: currentShopId,
        avito_shop_name: currentShopName
      };

      // Ищем ссылку на заказ
      const orderLink = cardElement.querySelector('a[href*="/orders/"]');
      if (orderLink) {
        baseData.order_url = orderLink.href;
        // Извлекаем ID заказа из URL
        const orderIdMatch = orderLink.href.match(/orders\/(\d+)/);
        if (orderIdMatch) {
          baseData.avito_order_id = orderIdMatch[1];
        }
      }

      // Парсим текст карточки
      const text = cardElement.textContent;

      // Номер заказа (№70000000362055374)
      const orderNumberMatch = text.match(/Заказ\s*№\s*(\d{14,20})/i);
      if (orderNumberMatch) {
        baseData.order_number = orderNumberMatch[1];
      }

      // Дата заказа парсится из API в fetchOrderDetails (orderCreated)

      // Определяем ТК через унифицированную систему
      baseData.delivery_company = detectDeliveryCompany(text);

      // Предварительный парсинг трека с карточки (будет уточнён через API)
      baseData.track_number = extractTrackFromText(text, baseData.delivery_company);

      // Если есть URL заказа - получаем детали через API
      // API возвращает МАССИВ товаров + уточнённые данные
      if (baseData.order_url) {
        const orderDetails = await fetchOrderDetails(baseData.order_url, baseData);

        if (orderDetails && orderDetails.items && orderDetails.items.length > 0) {

          return orderDetails.items.map((item, idx) => ({
            ...baseData,
            ...orderDetails.common,  // Общие данные (трек, покупатель)
            ...item,                  // Данные конкретного товара
            item_index: idx + 1,      // Номер товара в заказе
            items_total: orderDetails.items.length  // Всего товаров
          }));
        }
      }

      // Fallback - один товар
      return [baseData];

    } catch (error) {
      console.error('[AcreOne] Ошибка парсинга карточки:', error);
      return null;
    }
  }

  // Определение транспортной компании по тексту
  function detectDeliveryCompany(text) {
    for (const [key, config] of Object.entries(DELIVERY_COMPANIES)) {
      for (const name of config.names) {
        if (text.includes(name)) {
          // Возвращаем нормализованное имя
          return config.names[0];
        }
      }
    }
    return null;
  }

  // Получение ключа ТК по имени
  function getDeliveryKey(companyName) {
    if (!companyName) return null;
    for (const [key, config] of Object.entries(DELIVERY_COMPANIES)) {
      if (config.names.some(n => companyName.includes(n) || n.includes(companyName))) {
        return key;
      }
    }
    return null;
  }

  // Извлечение трека из текста с учётом ТК
  function extractTrackFromText(text, deliveryCompany) {
    const deliveryKey = getDeliveryKey(deliveryCompany);

    // Если знаем ТК - используем специфичные паттерны
    if (deliveryKey && DELIVERY_COMPANIES[deliveryKey]) {
      const config = DELIVERY_COMPANIES[deliveryKey];
      for (const pattern of config.trackPatterns) {
        const match = text.match(pattern);
        if (match && match[1]) {
          const track = match[1].replace(/[\s.]/g, '');
          if (track.length >= config.trackLength.min && track.length <= config.trackLength.max) {
            return track;
          }
        }
      }
    }

    // Универсальные паттерны
    // 5Post - номер заказа для QR (9 цифр: 091 009 013)
    // Если видим P-формат трека, ищем рядом номер заказа
    if (text.match(/5\s*post/i) || text.match(/P0\d\s*\d{3}\s*\d{4}/i)) {
      // Ищем номер заказа (9 цифр)
      const orderNumMatch = text.match(/(\d{3})\s+(\d{3})\s+(\d{3})/);
      if (orderNumMatch) {
        return orderNumMatch[1] + orderNumMatch[2] + orderNumMatch[3];
      }
    }

    // Яндекс (P + 9-12 цифр) - гибкий формат
    const yandexMatch = text.match(/P\d{2,3}[\s.]?\d{3}[\s.]?\d{3,4}[\s.]?\d{3,4}|P\d{9,12}/i);
    if (yandexMatch) {
      return yandexMatch[0].replace(/[\s.]/g, '');
    }

    // Почта России (14 цифр)
    const pochtaMatch = text.match(/\b(\d{3}\s?\d{3}\s?\d{3}\s?\d{5})\b/);
    if (pochtaMatch) {
      return pochtaMatch[1].replace(/\s/g, '');
    }

    // СДЭК (10-12 цифр)
    const cdekMatch = text.match(/\b(\d{3}\s?\d{3}\s?\d{4,6})\b/);
    if (cdekMatch) {
      const track = cdekMatch[1].replace(/\s/g, '');
      if (track.length >= 10 && track.length <= 12) {
        return track;
      }
    }

    // Авито (10 цифр)
    const avitoMatch = text.match(/\b(\d{3}\s?\d{3}\s?\d{4})\b/);
    if (avitoMatch) {
      return avitoMatch[1].replace(/\s/g, '');
    }

    // DPD (RU + 9 цифр, например RU115186992)
    const dpdMatch = text.match(/\b(RU\d{9})\b/i);
    if (dpdMatch) {
      return dpdMatch[1].toUpperCase();
    }

    // Dostavista / Курьер (Номер Достависты: 142 495 335)
    const dostavistaMatch = text.match(/Номер Достависты[:\s]*([0-9\s]{9,15})/i);
    if (dostavistaMatch) {
      return dostavistaMatch[1].replace(/\s/g, '');
    }

    return null;
  }

  // Извлечение трека из JSON API с учётом ТК
  function extractTrackFromApi(apiStr, deliveryCompany) {
    const deliveryKey = getDeliveryKey(deliveryCompany);

    // Специфичные паттерны для ТК
    if (deliveryKey && DELIVERY_COMPANIES[deliveryKey]) {
      const config = DELIVERY_COMPANIES[deliveryKey];
      for (const pattern of config.trackPatterns) {
        const match = apiStr.match(pattern);
        if (match && match[1]) {
          const track = match[1].replace(/[\s.]/g, '');
          return track;
        }
      }
    }

    // Универсальные паттерны
    for (const pattern of UNIVERSAL_TRACK_PATTERNS) {
      const match = apiStr.match(pattern);
      if (match && match[1]) {
        return match[1];
      }
    }

    return null;
  }

  // Получение деталей заказа (через API + HTML fallback)
  // Возвращает { common: {...}, items: [{...}, {...}] }
  async function fetchOrderDetails(orderUrl, baseData = {}) {
    try {

      // Извлекаем ID заказа из URL
      const orderIdMatch = orderUrl.match(/orders\/(\d+)/);
      const orderId = orderIdMatch ? orderIdMatch[1] : null;

      const result = {
        common: {},  // Общие данные заказа (трек, покупатель)
        items: []     // Массив товаров
      };

      // Сначала пробуем API для получения всех товаров
      if (orderId) {
        try {
          const apiUrl = `https://www.avito.ru/web/2/profile/order?referenceID=${orderId}&templateVersion=0`;
          const apiResponse = await fetch(apiUrl, { credentials: 'include' });
          if (apiResponse.ok) {
            const apiData = await apiResponse.json();
            const apiStr = JSON.stringify(apiData);

            // === ОБЩИЕ ДАННЫЕ ЗАКАЗА ===

            // Номер заказа из API (если не было с карточки)
            if (!baseData.order_number) {
              const orderNumMatch = apiStr.match(/"orderNumber":\s*"?(\d{14,20})"?/i) ||
                                    apiStr.match(/"referenceId":\s*"?(\d{14,20})"?/i);
              if (orderNumMatch) {
                result.common.order_number = orderNumMatch[1];
              }
            }

            // Дата создания заказа из API (orderCreated)
            const orderCreatedMatch = apiStr.match(/orderCreated":"([^"]+)"/i);
            if (orderCreatedMatch) {
              result.common.order_date = orderCreatedMatch[1];
            }

            // ТК из API
            const deliveryNameMatch = apiStr.match(/"deliveryProvider":\s*\{[^}]*"name":\s*\{[^}]*"nominative":\s*"([^"]+)"/);
            if (deliveryNameMatch) {
              result.common.delivery_company = deliveryNameMatch[1];
            } else if (apiStr.includes('avito-pvz') || apiStr.includes('avito_delivery')) {
              result.common.delivery_company = 'Авито';
            }

            // Трек-номер через унифицированную систему
            const deliveryCompany = result.common.delivery_company || baseData.delivery_company;
            const trackNumber = extractTrackFromApi(apiStr, deliveryCompany);
            if (trackNumber) {
              result.common.track_number = trackNumber;
            }

            // Покупатель
            const buyerNameMatch = apiStr.match(/"firstLetter":"[А-Яа-яA-Za-z]","name":"[^"]*?([А-Яа-яA-Za-z]{2,})[^"]*"/);
            if (buyerNameMatch) {
              result.common.buyer_name = buyerNameMatch[1];
            }

            // Channel ID для чата - собираем ВСЕ уникальные (для мульти-заказов)
            const allChannelIds = new Set();

            // 1. Ищем в URL-encoded формате (channelId%3D...)
            const urlEncodedMatches = apiStr.matchAll(/channelId[=%]3D([a-zA-Z0-9%~_-]+)/g);
            for (const match of urlEncodedMatches) {
              try {
                // Декодируем URL-encoded значение (например %7E -> ~)
                const decoded = decodeURIComponent(match[1]);
                allChannelIds.add(decoded);
              } catch (e) {
                allChannelIds.add(match[1]);
              }
            }

            // 2. Ищем в JSON формате ("channelId":"...")
            const jsonMatches = apiStr.matchAll(/"channelId":\s*"([^"]+)"/g);
            for (const match of jsonMatches) {
              allChannelIds.add(match[1]);
            }

            // 3. Ищем в формате u2i-~ (характерный для Авито)
            const u2iMatches = apiStr.matchAll(/u2i-[~]?[a-zA-Z0-9_-]+/g);
            for (const match of u2iMatches) {
              allChannelIds.add(match[0]);
            }

            // Очищаем channelIds от мусора (например &isMiniMessenger=true)
            const cleanedChannelIds = [...allChannelIds].map(id => {
              // Убираем всё после & (query параметры)
              const cleanId = id.split('&')[0];
              return cleanId;
            }).filter(id => {
              // Фильтруем валидные ID (должен начинаться с u2i- или u2I-)
              return id && id.length > 10 && /^u2[iI]-/.test(id);
            });

            // Убираем дубликаты после очистки
            const channelIdsArray = [...new Set(cleanedChannelIds)];

            if (channelIdsArray.length > 0) {
              // Сохраняем все channel_id для последующей проверки
              result.common.all_channel_ids = channelIdsArray;
              result.common.chat_channel_id = channelIdsArray[0]; // Первый как fallback
            }

            // === ДАННЫЕ МАГАЗИНА (ПРОДАВЦА) ===
            // Примечание: API заказа НЕ содержит sellerId/userId
            // Shop ID берётся из глобальной переменной currentShopId (извлекается из profile/extended)

            // Название магазина/продавца
            const sellerNameMatch = apiStr.match(/"sellerName":\s*"([^"]+)"/) ||
                                    apiStr.match(/"seller":\s*\{[^}]*"name":\s*"([^"]+)"/) ||
                                    apiStr.match(/"shop":\s*\{[^}]*"name":\s*"([^"]+)"/) ||
                                    apiStr.match(/"shopName":\s*"([^"]+)"/);
            if (sellerNameMatch) {
              result.common.shop_name = sellerNameMatch[1];
            }

            // Если не нашли в API - берём из глобальных переменных
            if (!result.common.shop_id && currentShopId) {
              result.common.shop_id = currentShopId;
            }
            if (!result.common.shop_name && currentShopName) {
              result.common.shop_name = currentShopName;
            }

            // === ПАРСИНГ МНОЖЕСТВЕННЫХ ТОВАРОВ ===
            // Ищем все товары в формате: "quantity":N,"title":"..."
            // Также ищем itemId, фото, цену для каждого
            const itemsData = parseMultipleItemsFromApi(apiStr);

            if (itemsData.length > 0) {
              result.items = itemsData;
            } else {
              // Fallback - один товар
              const singleItem = parseSingleItemFromApi(apiStr);
              if (singleItem) {
                result.items = [singleItem];
              }
            }

            // Убедимся что номер заказа есть в common (берём из baseData если не нашли в API)
            if (!result.common.order_number && baseData.order_number) {
              result.common.order_number = baseData.order_number;
            }
          } else {
            // API v2 вернул ошибку (403/500 для аккаунтов без ИП)
            // Пробуем альтернативный API v1 (работает для всех аккаунтов)
            try {
              const apiV1Url = `https://www.avito.ru/web/1/profile/order?orderId=${orderId}`;
              const apiV1Response = await fetch(apiV1Url, { credentials: 'include' });

              if (apiV1Response.ok) {
                const apiV1Data = await apiV1Response.json();
                const apiV1Str = JSON.stringify(apiV1Data);

                // Ищем 9-значный код для 5Post (формат: "value":"093744307"},"type":"copy")
                const dispatchMatch = apiV1Str.match(/"value":\s*"(\d{9})"\s*}\s*,\s*"type":\s*"copy"/);
                if (dispatchMatch) {
                  result.common.track_number = dispatchMatch[1];
                }

                // Создаём товар если его нет
                if (result.items.length === 0) {
                  result.items.push({});
                }

                // Ищем название товара (после "id":"item" ищем "value":"...")
                const itemTitleMatch = apiV1Str.match(/"id":\s*"item"[\s\S]*?"type":\s*"link"[\s\S]*?"value":\s*"([^"]+)"/);
                if (itemTitleMatch) {
                  result.items[0].title = itemTitleMatch[1];
                }

                // Ищем ссылку на товар
                const itemLinkMatch = apiV1Str.match(/"href":\s*"(https:\/\/avito\.ru\/\d+)"/);
                if (itemLinkMatch) {
                  result.items[0].item_url = itemLinkMatch[1];
                }

                // Ищем фото товара
                const photoMatch = apiV1Str.match(/"src":\s*"(https:\/\/\d+\.img\.avito\.st\/image\/[^"]+)"/);
                if (photoMatch) {
                  result.items[0].photo_url = photoMatch[1];
                }

                // Ищем цену (первое значение с ₽, которое не комиссия)
                const priceMatch = apiV1Str.match(/"value":\s*"([\d\s]+)\s*₽"/);
                if (priceMatch) {
                  result.items[0].price = parseInt(priceMatch[1].replace(/\s/g, ''));
                }

                // Ищем имя покупателя
                const buyerMatch = apiV1Str.match(/"value":"[^"]*?([А-Яа-яA-Za-z]+\s+[А-Яа-яA-Za-z]+)[^"]*?"[\s\S]{0,500}?отзыв/);
                if (buyerMatch) {
                  result.common.buyer_name = buyerMatch[1].trim();
                }

                // Ищем channel ID для чата (формат u2i-...)
                const channelMatch = apiV1Str.match(/u2[iI]-[~]?[a-zA-Z0-9_-]+/);
                if (channelMatch) {
                  result.common.chat_channel_id = channelMatch[0];
                  result.common.all_channel_ids = [channelMatch[0]];
                }

                // Устанавливаем ТК
                if (!result.common.delivery_company) {
                  result.common.delivery_company = baseData.delivery_company || '5Post';
                }
              }
            } catch (apiV1Err) {
              // Silent fail
            }
          }
        } catch (apiErr) {
          // Пробуем API v1 как fallback
          try {
            const apiV1Url = `https://www.avito.ru/web/1/profile/order?orderId=${orderId}`;
            const apiV1Response = await fetch(apiV1Url, { credentials: 'include' });

            if (apiV1Response.ok) {
              const apiV1Data = await apiV1Response.json();
              const apiV1Str = JSON.stringify(apiV1Data);

              // Ищем 9-значный код для 5Post
              const dispatchMatch = apiV1Str.match(/"value":\s*"(\d{9})"\s*}\s*,\s*"type":\s*"copy"/);
              if (dispatchMatch) {
                result.common.track_number = dispatchMatch[1];
                result.common.delivery_company = baseData.delivery_company || '5Post';
              }
            }
          } catch (apiV1Err) {
            // Silent fail
          }
        }
      }

      // === ПРИОРИТЕТ: Размер из чата, если нет - из описания ===

      // Флаг: найден ли размер в чате
      let sizeFoundInChat = false;

      // 1. Пробуем получить данные из чата (приоритетный источник для размера)
      // Для мульти-заказов проверяем ВСЕ чаты и находим тот где есть переписка
      let chatData = { sizes: [], articles: [] };
      const channelIds = result.common.all_channel_ids || (result.common.chat_channel_id ? [result.common.chat_channel_id] : []);

      if (channelIds.length > 0) {
        // Для мульти-заказов: проверяем все чаты и объединяем данные
        for (const channelId of channelIds) {
          const channelData = await fetchDataFromChat(channelId, result.items.length);

          // Если в этом чате нашли размеры - добавляем их
          if (channelData.sizes && channelData.sizes.length > 0) {
            channelData.sizes.forEach(size => {
              if (size && !chatData.sizes.includes(size)) {
                chatData.sizes.push(size);
              }
            });
          }

          // Аналогично для артикулов
          if (channelData.articles && channelData.articles.length > 0) {
            channelData.articles.forEach(article => {
              if (article && !chatData.articles.includes(article)) {
                chatData.articles.push(article);
              }
            });
          }

          // Если нашли достаточно данных - прекращаем поиск
          if (chatData.sizes.length >= result.items.length) {
            break;
          }
        }

        // AI анализ ВСЕГДА запускается ПРИНУДИТЕЛЬНО - он основной источник размера
        {
          // Пробуем AI на первом чате с данными
          for (const channelId of channelIds) {
            const chatAnalysis = await analyzeWithAI(channelId, result.items);
            if (chatAnalysis && chatAnalysis.sizes && chatAnalysis.sizes.some(s => s)) {
              // Применяем результаты AI к товарам
              for (let i = 0; i < result.items.length; i++) {
                if (chatAnalysis.sizes[i]) {
                  // AI - единственный источник размера (понимает контекст изменений)
                  const aiSize = chatAnalysis.sizes[i];
                  result.items[i].size = aiSize;
                  sizeFoundInChat = true;
                }
                if (chatAnalysis.articles && chatAnalysis.articles[i] && !result.items[i].article) {
                  result.items[i].article = chatAnalysis.articles[i];
                }
                if (chatAnalysis.colors && chatAnalysis.colors[i] && !result.items[i].color) {
                  result.items[i].color = chatAnalysis.colors[i];
                }
              }
              break; // Нашли данные через AI
            }
          }
        }

        // Артикулы из чата НЕ используем - только из объявления (там они точные)
      }

      // 2. Дополняем данные из объявлений (артикул, цвет, фото - НО НЕ РАЗМЕР!)
      // РАЗМЕР БЕРЁМ ТОЛЬКО ИЗ ЧАТА!
      for (let i = 0; i < result.items.length; i++) {
        const item = result.items[i];

        // Определяем URL объявления (из avito_item_id или напрямую из item_url)
        let itemUrl = null;
        if (item.avito_item_id) {
          itemUrl = `https://www.avito.ru/items/${item.avito_item_id}`;
        } else if (item.item_url) {
          // Для API v1 fallback - item_url уже содержит ссылку
          itemUrl = item.item_url;
          // Преобразуем короткий URL в полный если нужно
          if (itemUrl.startsWith('https://avito.ru/')) {
            itemUrl = itemUrl.replace('https://avito.ru/', 'https://www.avito.ru/');
          }
        }

        // Получаем дополнительные данные из объявления
        if (itemUrl) {
          try {
            const itemData = await fetchItemData(itemUrl);
            // Артикул - ВСЕГДА берём из объявления (он точный, в чате может быть шум)
            // Цвет - берём из объявления если нет в чате
            // РАЗМЕР НЕ БЕРЁМ ИЗ ОБЪЯВЛЕНИЯ НИКОГДА! Только из чата!
            if (itemData.article && !item.article) item.article = itemData.article;
            if (itemData.color && !item.color) item.color = itemData.color;
            // Фото и URL всегда берём из объявления (если ещё нет)
            if (itemData.photo_url && !item.photo_url) item.photo_url = itemData.photo_url;
            if (itemData.photo_url_2) item.photo_url_2 = itemData.photo_url_2;
            if (itemData.item_url) item.item_url = itemData.item_url;
          } catch (itemErr) {
            // Silent fail
          }
        }
      }

      return result;

    } catch (error) {
      console.error('[AcreOne] Ошибка загрузки деталей:', error);
      return { common: {}, items: [] };
    }
  }

  // Извлечение фото из JSON API
  function extractPhotoFromApi(apiStr, context = null) {
    const searchStr = context || apiStr;

    // Паттерны для фото (от лучшего к худшему)
    const photoPatterns = [
      // Стандартный формат avito.st
      /"src":\s*"(https:\/\/\d+\.img\.avito\.st\/image\/\d+\/[^"]+)"/i,
      // Альтернативный формат
      /"image":\s*"(https:\/\/[^"]+\.avito\.st[^"]+)"/i,
      // URL в любом поле с avito.st
      /"[^"]*url[^"]*":\s*"(https:\/\/[^"]+\.avito\.st\/image[^"]+)"/i,
      // Любое изображение avito
      /(https:\/\/\d+\.img\.avito\.st\/image\/\d+\/\d+x\d+\/[^"'\s]+)/i,
      // Fallback - любой URL с img.avito.st
      /(https:\/\/\d+\.img\.avito\.st[^"'\s]+)/i
    ];

    for (const pattern of photoPatterns) {
      const match = searchStr.match(pattern);
      if (match && match[1]) {
        // Увеличиваем размер фото если возможно
        let photoUrl = match[1];
        // Заменяем маленький размер на большой
        photoUrl = photoUrl.replace(/\/\d+x\d+\//, '/640x480/');
        return photoUrl;
      }
    }

    return null;
  }

  // Парсинг множественных товаров из API
  function parseMultipleItemsFromApi(apiStr) {
    const items = [];

    try {
      // Ищем паттерн товаров: "quantity":N,"title":"..."
      const itemRegex = /"quantity":(\d+),"title":"([^"]+)"/g;
      let match;

      while ((match = itemRegex.exec(apiStr)) !== null) {
        const quantity = parseInt(match[1]);
        const title = match[2];

        // Ищем данные рядом с этим товаром (в пределах 1000 символов)
        const contextStart = Math.max(0, match.index - 1000);
        const contextEnd = Math.min(apiStr.length, match.index + 1000);
        const context = apiStr.substring(contextStart, contextEnd);

        const itemIdMatch = context.match(/itemId[="](\d{9,12})/);
        const priceMatch = context.match(/"price":(\d+)/);
        const photoUrl = extractPhotoFromApi(apiStr, context);

        // Создаём записи для каждой единицы товара
        for (let i = 0; i < quantity; i++) {
          items.push({
            item_title: title,
            avito_item_id: itemIdMatch ? itemIdMatch[1] : null,
            item_price: priceMatch ? parseInt(priceMatch[1]) : null,
            photo_url: photoUrl,
            quantity: 1,
            quantity_in_order: quantity,
            size: null,
            article: null,
            color: null
          });
        }
      }
    } catch (e) {
      console.error('[AcreOne] Ошибка парсинга множественных товаров:', e);
    }

    return items;
  }

  // Парсинг одного товара из API (fallback)
  function parseSingleItemFromApi(apiStr) {
    try {
      const titleMatch = apiStr.match(/"quantity":\d+,"title":"([^"]+)"/);
      const itemIdMatch = apiStr.match(/itemId[="](\d{9,12})/);
      const priceMatch = apiStr.match(/"price":(\d+)/);
      const photoUrl = extractPhotoFromApi(apiStr);

      if (titleMatch) {
        return {
          item_title: titleMatch[1],
          avito_item_id: itemIdMatch ? itemIdMatch[1] : null,
          item_price: priceMatch ? parseInt(priceMatch[1]) : null,
          photo_url: photoUrl,
          quantity: 1,
          size: null,
          article: null,
          color: null
        };
      }
    } catch (e) {
      console.error('[AcreOne] Ошибка парсинга одиночного товара:', e);
    }

    return null;
  }

  // Получение данных из чата (размер, артикул) через regex
  // v2.0 - Улучшенная логика: ищем ПОСЛЕДНИЙ подтверждённый размер
  async function fetchDataFromChat(channelId, itemsCount = 1) {
    if (!channelId) return { sizes: [], articles: [] };

    try {
      const response = await fetch('https://www.avito.ru/web/1/messenger/getUserVisibleMessages', {
        method: 'POST',
        credentials: 'include',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ channelId, limit: 100, order: 0 })
      });

      if (!response.ok) {
        return { sizes: [], articles: [] };
      }

      const data = await response.json();

      // Нормализация кириллических букв в латинские для размеров
      const normalizeSizeLetter = (str) => {
        return str
          .replace(/[МмМ]/g, 'M')  // Кириллическая М -> M
          .replace(/[Сс]/g, 'S')   // Кириллическая С -> S
          .replace(/[ХхХ]/g, 'X')  // Кириллическая Х -> X
          .replace(/[Ll]/g, 'L');  // L остаётся L
      };

      // Парсим сообщения из JSON
      const chatMessages = data.success?.messages || data.messages || data.result?.messages || [];

      // Собираем размеры с приоритетом (чем выше приоритет - тем важнее)
      // Приоритет: зафиксировали/фиксирую (3) > беру/хочу/давайте (2) > просто "размер X" (1)
      const foundSizes = []; // { size, priority, index }

      for (let i = 0; i < chatMessages.length; i++) {
        const msg = chatMessages[i];

        let text = msg.body?.text?.text || msg.content?.text || msg.text || msg.body?.text || '';
        if (!text || typeof text !== 'string') continue;

        const normalizedText = normalizeSizeLetter(text);

        // Пропускаем сообщения которые выглядят как перечисление размеров (автоответы бота)
        if (/размер[ыа]?\s*[:.]?\s*(XXS|XS|S|M|L|XL|XXL|XXXL)(\s*[,\/]\s*(XXS|XS|S|M|L|XL|XXL|XXXL)){2,}/i.test(normalizedText)) {
          continue;
        }

        // Приоритет 3: "зафиксировали/фиксирую размер X" - финальное подтверждение
        // Учитываем: "зафиксировал размер M", "зафиксировал для вас размер M", "зафиксировали ваш размер L"
        const fixMatch = normalizedText.match(/(?:зафиксировал[иа]?|фиксирую|фиксируем)[^а-яa-z]*(?:для\s+вас\s+|ваш\s+)?размер[:\s]*(XXS|XS|S|M|L|XL|XXL|XXXL|2XL|3XL|\d{2})/i);
        if (fixMatch) {
          const size = fixMatch[1].toUpperCase().replace('2XL', 'XXL').replace('3XL', 'XXXL');
          foundSizes.push({ size, priority: 3, index: i });
          continue;
        }

        // Приоритет 2: "беру/хочу/давайте/тогда X" - выбор покупателя
        const choiceMatch = normalizedText.match(/(?:беру|хочу|давайте|давай|буду|возьму|нужен|нужна|закажу|заказываю|тогда|тогда\s+наверное|ну\s+тогда|значит|понял|поняла|окей|ок,?\s*)[:\s!,]*(XXS|XS|S|M|L|XL|XXL|XXXL|2XL|3XL|\d{2})/i);
        if (choiceMatch) {
          const size = choiceMatch[1].toUpperCase().replace('2XL', 'XXL').replace('3XL', 'XXXL');
          foundSizes.push({ size, priority: 2, index: i });
          continue;
        }

        // Приоритет 2: "мне L" или "мне нужен L" - прямой выбор
        const directMatch = normalizedText.match(/мне\s+(?:нужен\s+|нужна\s+)?(XXS|XS|S|M|L|XL|XXL|XXXL|2XL|3XL|\d{2})(?:\s|$|[.,!?])/i);
        if (directMatch) {
          const size = directMatch[1].toUpperCase().replace('2XL', 'XXL').replace('3XL', 'XXXL');
          foundSizes.push({ size, priority: 2, index: i });
          continue;
        }

        // Приоритет 1: "размер X" - упоминание размера
        if (!/есть\s+(?:ли\s+)?(?:у\s+вас\s+)?размер/i.test(normalizedText) &&
            !/размер[ыа]?\s+в\s+наличии/i.test(normalizedText)) {
          const sizeMatch = normalizedText.match(/размер[:\s]*(XXS|XS|S|M|L|XL|XXL|XXXL|2XL|3XL|\d{2})(?:\s|$|[.,!?])/i);
          if (sizeMatch) {
            const size = sizeMatch[1].toUpperCase().replace('2XL', 'XXL').replace('3XL', 'XXXL');
            foundSizes.push({ size, priority: 1, index: i });
            continue;
          }
        }

        // Standalone размер в коротком сообщении (< 30 символов)
        if (normalizedText.length < 30) {
          const exactMatch = normalizedText.match(/^[^a-zA-Zа-яА-Я]*(XXS|XS|S|M|L|XL|XXL|XXXL|2XL|3XL)[^a-zA-Zа-яА-Я]*$/i);
          if (exactMatch) {
            const size = exactMatch[1].toUpperCase().replace('2XL', 'XXL').replace('3XL', 'XXXL');
            foundSizes.push({ size, priority: 2, index: i });
          } else {
            const endMatch = normalizedText.match(/(?:ну|да|ага|хорошо|ладно|ок)\s*,?\s*(XXS|XS|S|M|L|XL|XXL|XXXL|2XL|3XL)\s*[!.?]*$/i);
            if (endMatch) {
              const size = endMatch[1].toUpperCase().replace('2XL', 'XXL').replace('3XL', 'XXXL');
              foundSizes.push({ size, priority: 2, index: i });
            }
          }
        }
      }

      // Выбираем размер: сначала по приоритету, потом по позиции (последний)
      foundSizes.sort((a, b) => {
        if (b.priority !== a.priority) return b.priority - a.priority;
        return b.index - a.index;
      });

      const allSizes = [];
      if (foundSizes.length > 0) {
        const bestSize = foundSizes[0].size;
        allSizes.push(bestSize);
      }

      // Артикулы из чата НЕ парсим - берём ТОЛЬКО из объявления (там они точные)
      return {
        sizes: allSizes,
        articles: [],
        size: allSizes[0] || null,
        article: null
      };

    } catch (error) {
      console.error('[AcreOne] Ошибка загрузки чата:', error);
      return { sizes: [], articles: [] };
    }
  }

  // AI анализ чата для извлечения размеров, артикулов, цветов
  // ПРИНУДИТЕЛЬНО ВКЛЮЧЕН - основной источник размера
  async function analyzeWithAI(channelId, items) {
    if (!channelId) return null;
    // useAiAnalysis больше не проверяем - AI всегда включен

    try {
      // Получаем сообщения чата
      const chatResponse = await fetch('https://www.avito.ru/web/1/messenger/getUserVisibleMessages', {
        method: 'POST',
        credentials: 'include',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ channelId, limit: 100, order: 0 })
      });

      if (!chatResponse.ok) return null;

      const chatData = await chatResponse.json();

      // Извлекаем текст сообщений (API возвращает в success.messages)
      const messages = [];
      const chatMessages = chatData.success?.messages || chatData.messages || chatData.result?.messages || [];

      for (const msg of chatMessages) {
        const text = msg.body?.text?.text || msg.content?.text || msg.text || msg.body?.text || '';
        if (text && text.trim()) {
          messages.push(text.trim());
        }
      }

      // ВАЖНО: сообщения приходят от новых к старым, переворачиваем для хронологии
      messages.reverse();

      if (messages.length === 0) return null;

      // Отправляем на бэкенд для AI анализа
      const { acreone_api_url, acreone_token } = await chrome.storage.local.get([
        'acreone_api_url',
        'acreone_token'
      ]);

      if (!acreone_token) {
        return null;
      }

      const apiUrl = acreone_api_url || 'https://api.acreone.ru';
      const requestBody = {
        messages: messages.join('\n'),
        items: items.map(i => ({
          title: i.item_title,
          quantity: i.quantity_in_order || 1
        }))
      };

      const response = await fetch(`${apiUrl}/rest/orders/analyze-chat`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${acreone_token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(requestBody)
      });

      if (!response.ok) {
        return null;
      }

      const result = await response.json();

      return result;

    } catch (error) {
      console.error('[AcreOne] Ошибка AI анализа:', error);
      return null;
    }
  }


  // Получение данных объявления (артикул, фото)
  async function fetchItemData(itemUrl) {
    if (!itemUrl) return {};

    try {
      const response = await fetch(itemUrl, { credentials: 'include' });

      // Сохраняем финальный URL (после редиректа)
      const finalUrl = response.url;

      const html = await response.text();
      const parser = new DOMParser();
      const doc = parser.parseFromString(html, 'text/html');

      const data = {
        item_url: finalUrl
      };

      const bodyText = doc.body.textContent || doc.body.innerText;

      // === АРТИКУЛ ===
      // Расширенные паттерны для артикулов
      const articlePatterns = [
        // Стандартные форматы
        /Артикул[:\s]+([A-Za-zА-Яа-я0-9_\-]{2,20})/i,
        /Арт[.:\s]+([A-Za-zА-Яа-я0-9_\-]{2,20})/i,
        /Art[.:\s]+([A-Za-z0-9_\-]{2,20})/i,
        /SKU[:\s]+([A-Za-z0-9_\-]{2,20})/i,
        /Код[:\s]+([A-Za-zА-Яа-я0-9_\-]{2,15})/i,
        // Форматы типа "ЛЧ_124", "TWV-030", "ABC123"
        /(?:^|\s)([A-ZА-Я]{1,4}[_\-]?\d{2,5})(?:\s|$)/i,
        // Форматы типа "123-ABC"
        /(?:^|\s)(\d{2,5}[_\-][A-ZА-Я]{1,4})(?:\s|$)/i
      ];

      for (const pattern of articlePatterns) {
        const match = bodyText.match(pattern);
        if (match && match[1]) {
          const article = match[1].trim();
          // Проверяем что это не мусор
          if (article.length >= 2 && article.length <= 20) {
            // Не кириллические слова типа "Характеристики", "Описание"
            if (!article.match(/^[А-Яа-я]{5,}$/)) {
              data.article = article;
              break;
            }
          }
        }
      }

      // === РАЗМЕР ===
      // УБРАНО! Размер из описания объявления НЕ ИЗВЛЕКАЕМ!
      // Размер должен определяться ТОЛЬКО из переписки с покупателем
      // Иначе получаем неверные значения когда покупатель просит другой размер

      // === ФОТО (до 2 штук в полном качестве) ===
      // v1.10.8: Улучшен парсинг - ищем крупные фото вместо превьюшек
      // Авито URL формат: https://XX.img.avito.st/image/1/1.{HASH1}.{HASH2}?cqp={PARAMS}
      // HASH1 содержит индикатор размера: a3 = превью (75x55), a4 = полный размер (722x960)
      const photos = [];
      const seenPhotoIds = new Set();

      // Функция извлечения уникального ID фото из URL
      const getPhotoId = (url) => {
        if (!url || !url.includes('avito.st')) return null;
        // Новый формат: /image/1/1.HASH1.HASH2
        const hashMatch = url.match(/\/image\/\d+\/\d+\.([a-zA-Z0-9_-]+)/);
        if (hashMatch) {
          // Первые 5 символов = уникальный ID фото (без индикатора размера)
          return hashMatch[1].substring(0, 5);
        }
        return null;
      };

      // Функция определения качества URL (чем выше - тем лучше)
      const getPhotoQuality = (url) => {
        if (!url) return 0;
        // Признаки полноразмерного фото
        if (url.includes('a4g5') || url.includes('DhMA') || url.includes('a4')) return 100;
        // Признаки среднего качества
        if (url.includes('640x480') || url.includes('a3g4')) return 50;
        // Превью
        if (url.includes('a3') || url.includes('75x55')) return 10;
        return 30; // неизвестный формат
      };

      // Функция трансформации URL в полноразмерный
      const getLargePhotoUrl = (url) => {
        if (!url) return url;
        // Для старого формата заменяем размер
        url = url.replace(/\/\d+x\d+\//, '/1280x960/');
        return url;
      };

      // Собираем URL фото из ГАЛЕРЕИ товара (не из всего HTML, чтобы не захватить аватарки)
      const galleryPhotoUrls = new Map(); // photoId -> {url, quality}

      // ID фото которые точно НЕ относятся к товару (аватарки, баннеры)
      const excludedPhotoIds = new Set(['ob2T9', 'tVa8N', '6rYL3']);

      // 1. Из галереи (DOM) - основной источник
      const galleryImgs = doc.querySelectorAll('[class*="gallery"] img, [class*="Gallery"] img, [class*="slider"] img, [class*="image-frame"] img, [data-marker="image-frame/image"]');
      galleryImgs.forEach(img => {
        const src = img.src || img.getAttribute('data-src');
        if (src && src.includes('avito.st')) {
          const photoId = getPhotoId(src);
          const quality = getPhotoQuality(src);
          if (photoId && !excludedPhotoIds.has(photoId)) {
            const existing = galleryPhotoUrls.get(photoId);
            if (!existing || existing.quality < quality) {
              galleryPhotoUrls.set(photoId, { url: src, quality, source: 'gallery' });
            }
          }
        }
      });

      // 2. Из JSON __NEXT_DATA__ (images массив товара) - надёжный источник
      const jsonDataMatch = html.match(/"images"\s*:\s*\[([^\]]+)\]/);
      if (jsonDataMatch) {
        const imageUrls = [...jsonDataMatch[1].matchAll(/"(https:\/\/[^"]+avito\.st[^"]+)"/g)];
        for (const imgMatch of imageUrls) {
          const url = imgMatch[1];
          const photoId = getPhotoId(url);
          const quality = getPhotoQuality(url);
          if (photoId && !excludedPhotoIds.has(photoId)) {
            const existing = galleryPhotoUrls.get(photoId);
            if (!existing || existing.quality < quality) {
              galleryPhotoUrls.set(photoId, { url, quality, source: 'json' });
            }
          }
        }
      }

      // Сортируем по качеству и берём лучшие URL
      const sortedPhotos = [...galleryPhotoUrls.entries()]
        .sort((a, b) => b[1].quality - a[1].quality)
        .slice(0, 2);

      for (const [photoId, data] of sortedPhotos) {
        if (!seenPhotoIds.has(photoId)) {
          seenPhotoIds.add(photoId);
          photos.push(getLargePhotoUrl(data.url));
        }
      }

      // 3. Meta og:image как fallback
      if (photos.length === 0) {
        const ogImage = doc.querySelector('meta[property="og:image"]');
        if (ogImage?.content && ogImage.content.includes('avito.st')) {
          photos.push(getLargePhotoUrl(ogImage.content));
        }
      }

      // Если нашли только 1 фото - пробуем найти второе в JSON images товара
      if (photos.length === 1 && finalUrl && !finalUrl.includes('#extended')) {
        try {
          // Пробуем найти images массив в JSON
          const imagesArrayMatch = html.match(/"images"\s*:\s*(\[[^\]]+\])/);
          if (imagesArrayMatch) {
            const imagesJson = imagesArrayMatch[1];
            const allImageUrls = [...imagesJson.matchAll(/"(https:\/\/[^"]+avito\.st[^"]+)"/g)];

            for (const imgMatch of allImageUrls) {
              if (photos.length >= 2) break;
              const url = imgMatch[1];
              const photoId = getPhotoId(url);
              if (photoId && !excludedPhotoIds.has(photoId) && !seenPhotoIds.has(photoId)) {
                seenPhotoIds.add(photoId);
                photos.push(getLargePhotoUrl(url));
              }
            }
          }
        } catch (extError) {
          // Игнорируем ошибку поиска доп. фото
        }
      }

      // Сохраняем найденные фото
      if (photos.length > 0) {
        data.photo_url = photos[0];
      }
      if (photos.length > 1) {
        data.photo_url_2 = photos[1];
      }

      // === ЦВЕТ ===
      // Список известных цветов для точного сопоставления
      const KNOWN_COLORS = [
        'белый', 'черный', 'красный', 'синий', 'зеленый', 'желтый', 'оранжевый',
        'розовый', 'фиолетовый', 'серый', 'коричневый', 'бежевый', 'голубой',
        'бордовый', 'хаки', 'молочный', 'кремовый', 'персиковый', 'мятный',
        'лавандовый', 'пудровый', 'графитовый', 'антрацит', 'слоновая кость',
        'white', 'black', 'red', 'blue', 'green', 'yellow', 'orange', 'pink',
        'purple', 'gray', 'grey', 'brown', 'beige', 'navy', 'maroon', 'khaki'
      ];

      const colorPatterns = [
        /Цвет[:\s]+([А-ЯЁа-яёA-Za-z]+)/i,
        /Color[:\s]+([A-Za-z]+)/i,
        // Авито HTML: Цвет<!-- -->: </span>Чёрный</li>
        /Цвет[\s\S]*?<\/span>([А-ЯЁа-яёA-Za-z]+)<\/li>/i,
        // JSON в сыром HTML: "Цвет":"Чёрный"
        /"Цвет"\s*:\s*"([А-ЯЁа-яёA-Za-z]+)"/i
      ];

      // Ищем сначала в bodyText, потом в сыром HTML
      for (const pattern of colorPatterns) {
        let match = bodyText.match(pattern);
        // Если не нашли в bodyText, ищем в сыром HTML
        if (!match) {
          match = html.match(pattern);
        }
        if (match && match[1]) {
          // Берём только первое слово - до заглавной буквы или цифры
          let color = match[1].trim();

          // Фильтр CSS-значений (rgba, rgb, var, url, px, etc.)
          const cssKeywords = ['rgba', 'rgb', 'var', 'url', 'calc', 'inherit', 'transparent', 'none'];
          if (cssKeywords.includes(color.toLowerCase())) {
            continue;
          }

          // Разбиваем по заглавным буквам (БелыйПоднять -> Белый)
          const camelCaseSplit = color.match(/^[А-ЯЁа-яёA-Za-z][а-яёa-z]*/);
          // Применяем split только если результат >= 3 символов
          if (camelCaseSplit && camelCaseSplit[0].length >= 3) {
            color = camelCaseSplit[0];
          }
          // Проверяем минимальную длину (не "Ч", а "Черный")
          if (color.length >= 3 && color.length <= 20) {
            // Проверяем что это похоже на цвет
            const colorLower = color.toLowerCase();
            const isKnownColor = KNOWN_COLORS.some(c => colorLower.includes(c) || c.includes(colorLower));
            if (isKnownColor || color.length >= 4) {
              data.color = color;
              break;
            }
          }
        }
      }

      // Характеристики (параметры объявления)
      const params = doc.querySelectorAll('[class*="param"], [class*="Param"], [class*="specs"] li, [class*="item-params"] li');
      params.forEach(param => {
        const text = param.textContent || '';
        const lowerText = text.toLowerCase();

        if (!data.color && lowerText.includes('цвет')) {
          const colorMatch = text.match(/Цвет[:\s]*([А-ЯЁа-яёA-Za-z]+)/i);
          if (colorMatch && colorMatch[1].length >= 3) {
            data.color = colorMatch[1];
          }
        }
        // Размер из параметров объявления УБРАН - берём ТОЛЬКО из чата
        // if (!data.size && lowerText.includes('размер')) { ... }
        if (!data.article && (lowerText.includes('артикул') || lowerText.includes('код'))) {
          const artMatch = text.match(/(?:Артикул|Код)[:\s]*([A-Za-zА-Яа-я0-9_\-]{2,20})/i);
          if (artMatch) {
            data.article = artMatch[1];
          }
        }
      });

      return data;

    } catch (error) {
      console.error('[AcreOne] Ошибка загрузки объявления:', error);
      return {};
    }
  }

  // Отрисовка панели с заказами (с группировкой мульти-заказов)
  function renderOrdersPanel() {
    const listEl = document.getElementById('acreone-orders-list');
    const sendBtn = document.getElementById('acreone-send-btn');

    // Показываем кнопку отправки (могла быть скрыта после ошибки)
    if (sendBtn) {
      sendBtn.style.display = 'block';
    }

    if (collectedOrders.length === 0) {
      listEl.innerHTML = '<p style="color: #9ca3af; text-align: center;">Заказы не найдены</p>';
      if (sendBtn) sendBtn.disabled = true;
      return;
    }

    // SECURITY: Функция для безопасного URL (только https или data:)
    const safeUrl = (url) => {
      if (!url) return 'data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 width=%2250%22 height=%2250%22><rect fill=%22%23333%22 width=%2250%22 height=%2250%22/></svg>';
      const str = String(url);
      if (str.startsWith('https://') || str.startsWith('data:')) return str;
      return 'data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 width=%2250%22 height=%2250%22><rect fill=%22%23333%22 width=%2250%22 height=%2250%22/></svg>';
    };

    // Блок информации о магазине (SECURITY: escapeHtml)
    const shopInfoHtml = currentShopId ? `
      <div class="acreone-shop-info">
        <strong>${escapeHtml(currentShopName || 'Магазин')}</strong>
        <span style="color: #6b7280; font-size: 11px; margin-left: 8px;">ID: ${escapeHtml(currentShopId)}</span>
      </div>
    ` : '';

    // Группируем заказы по order_number или avito_order_id
    const groupedOrders = {};
    collectedOrders.forEach((order, index) => {
      const groupKey = order.order_number || order.avito_order_id || `single_${index}`;
      if (!groupedOrders[groupKey]) {
        groupedOrders[groupKey] = [];
      }
      groupedOrders[groupKey].push({ ...order, originalIndex: index });
    });

    // SVG иконки
    const iconMissing = `<svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="#f87171" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>`;

    // Генерируем HTML для каждой группы
    const ordersHtml = Object.entries(groupedOrders).map(([groupKey, items]) => {
      const firstItem = items[0];
      const isMulti = items.length > 1;

      if (isMulti) {
        // Мульти-заказ - группируем под общей шапкой (SECURITY: escapeHtml для всех данных)
        const itemsHtml = items.map((order, idx) => {
          const isLast = idx === items.length - 1;
          return `
            <div class="acreone-order-item ${isLast ? 'last' : ''}" data-index="${order.originalIndex}">
              <div class="acreone-order-item-connector">
                <span class="acreone-connector-line ${isLast ? 'last' : ''}"></span>
              </div>
              <input type="checkbox" checked data-index="${order.originalIndex}" class="acreone-order-select">
              <div class="acreone-photo-wrapper${order.photo_url_2 ? ' has-extra' : ''}">
                ${order.photo_url_2 ? '<div class="acreone-photo-extra">+1</div>' : ''}
                <img src="${safeUrl(order.photo_url)}" alt="" class="acreone-order-item-img">
              </div>
              <div class="acreone-order-item-info">
                <div class="acreone-order-item-title">${escapeHtml(order.item_title || 'Без названия')}</div>
                <div class="acreone-order-item-meta">
                  <span class="acreone-editable-size ${!order.size ? 'missing' : ''}" data-index="${order.originalIndex}" title="Клик для редактирования">${escapeHtml(order.size || 'без размера')}</span>
                  <span class="acreone-separator">•</span>
                  <span class="${!order.article ? 'missing' : ''}">${escapeHtml(order.article || 'без артикула')}</span>
                </div>
              </div>
            </div>
          `;
        }).join('');

        return `
          <div class="acreone-order-group" data-order="${escapeHtml(groupKey)}">
            <div class="acreone-order-group-header">
              <div class="acreone-order-group-info">
                <div class="acreone-order-group-number">№${escapeHtml(firstItem.order_number || groupKey)}${firstItem.order_date ? ` <span class="acreone-order-date">от ${escapeHtml(firstItem.order_date)}</span>` : ''}</div>
                <div class="acreone-order-group-meta">
                  <span class="acreone-order-group-count">${items.length} товара</span>
                  <span class="acreone-separator">•</span>
                  <span>${escapeHtml(firstItem.buyer_name || 'Покупатель')}</span>
                  ${firstItem.avito_shop_name ? `<span class="acreone-separator">•</span><span>${escapeHtml(firstItem.avito_shop_name)}</span>` : ''}
                </div>
              </div>
              <div class="acreone-order-group-delivery">
                <span>${escapeHtml(firstItem.delivery_company || '—')}</span>
                <span class="acreone-order-group-track">${escapeHtml(firstItem.track_number || 'нет трека')}</span>
              </div>
            </div>
            <div class="acreone-order-group-items">
              ${itemsHtml}
            </div>
          </div>
        `;
      } else {
        // Одиночный заказ - обычная карточка (SECURITY: escapeHtml для всех данных)
        const order = firstItem;
        return `
          <div class="acreone-order-card" data-index="${order.originalIndex}">
            <div class="acreone-order-card-header">
              <input type="checkbox" checked data-index="${order.originalIndex}" class="acreone-order-select">
              <div class="acreone-photo-wrapper${order.photo_url_2 ? ' has-extra' : ''}">
                ${order.photo_url_2 ? '<div class="acreone-photo-extra">+1</div>' : ''}
                <img src="${safeUrl(order.photo_url)}" alt="">
              </div>
              <div class="acreone-order-card-info">
                <div class="acreone-order-card-title">${escapeHtml(order.item_title || 'Без названия')}</div>
                <div class="acreone-order-card-details">${escapeHtml(order.delivery_company || '—')} • ${escapeHtml(order.track_number || 'нет трека')}</div>
                ${order.order_number ? `<div class="acreone-order-card-order-num">№${escapeHtml(order.order_number)}${order.order_date ? ` <span class="acreone-order-date">от ${escapeHtml(order.order_date)}</span>` : ''}</div>` : ''}
              </div>
            </div>
            <div class="acreone-order-card-row">
              <span class="acreone-order-card-label">Артикул</span>
              <span class="acreone-order-card-value acreone-editable-article ${!order.article ? 'missing' : ''}" data-index="${order.originalIndex}" title="Клик для редактирования">${order.article ? escapeHtml(order.article) : `${iconMissing} нет`}</span>
            </div>
            <div class="acreone-order-card-row">
              <span class="acreone-order-card-label">Размер</span>
              <span class="acreone-order-card-value acreone-editable-size ${!order.size ? 'missing' : ''}" data-index="${order.originalIndex}" title="Клик для редактирования">${order.size ? escapeHtml(order.size) : `${iconMissing} нет`}</span>
            </div>
            <div class="acreone-order-card-row">
              <span class="acreone-order-card-label">Цвет</span>
              <span class="acreone-order-card-value acreone-editable-color ${!order.color ? 'missing' : ''}" data-index="${order.originalIndex}" title="Клик для редактирования">${order.color ? escapeHtml(order.color) : `${iconMissing} нет`}</span>
            </div>
            <div class="acreone-order-card-row">
              <span class="acreone-order-card-label">Покупатель</span>
              <span class="acreone-order-card-value">${escapeHtml(order.buyer_name || '—')}</span>
            </div>
            ${order.avito_shop_name ? `<div class="acreone-order-card-row"><span class="acreone-order-card-label">Магазин</span><span class="acreone-order-card-value">${escapeHtml(order.avito_shop_name)}</span></div>` : ''}
          </div>
        `;
      }
    }).join('');

    listEl.innerHTML = shopInfoHtml + ordersHtml;

    // Обработчики чекбоксов
    listEl.querySelectorAll('.acreone-order-select').forEach(cb => {
      cb.addEventListener('change', updateSendButton);
    });

    // Обработчики редактирования размера
    listEl.querySelectorAll('.acreone-editable-size').forEach(el => {
      el.addEventListener('click', (e) => {
        e.stopPropagation();
        const index = parseInt(el.dataset.index);
        const order = collectedOrders[index];
        if (!order) return;

        // Создаём инпут
        const currentSize = order.size || '';
        const input = document.createElement('input');
        input.type = 'text';
        input.value = currentSize;
        input.className = 'acreone-size-input';
        input.placeholder = 'Размер';
        input.style.cssText = 'width: 60px; padding: 2px 6px; border: 1px solid #f97316; border-radius: 4px; background: #1a1a1f; color: #fff; font-size: 12px; outline: none;';

        // Заменяем span на input
        el.replaceWith(input);
        input.focus();
        input.select();

        // Сохранение при потере фокуса или Enter
        const saveSize = () => {
          const newSize = input.value.trim().toUpperCase();
          collectedOrders[index].size = newSize || null;

          // Создаём новый span
          const newSpan = document.createElement('span');
          newSpan.className = `acreone-editable-size ${!newSize ? 'missing' : ''}`;
          newSpan.dataset.index = index;
          newSpan.title = 'Клик для редактирования';
          newSpan.textContent = newSize || 'без размера';
          newSpan.style.cursor = 'pointer';

          input.replaceWith(newSpan);

          // Добавляем обработчик заново
          newSpan.addEventListener('click', (e) => {
            e.stopPropagation();
            newSpan.click();
          });

          // Перерендериваем панель для корректной работы
          renderOrdersPanel();
        };

        input.addEventListener('blur', saveSize);
        input.addEventListener('keydown', (e) => {
          if (e.key === 'Enter') {
            e.preventDefault();
            input.blur();
          }
          if (e.key === 'Escape') {
            input.value = currentSize;
            input.blur();
          }
        });
      });
      el.style.cursor = 'pointer';
    });

    // Обработчики редактирования артикула
    listEl.querySelectorAll('.acreone-editable-article').forEach(el => {
      el.addEventListener('click', (e) => {
        e.stopPropagation();
        const index = parseInt(el.dataset.index);
        const order = collectedOrders[index];
        if (!order) return;

        const currentArticle = order.article || '';
        const input = document.createElement('input');
        input.type = 'text';
        input.value = currentArticle;
        input.className = 'acreone-article-input';
        input.placeholder = 'Артикул';
        input.style.cssText = 'width: 80px; padding: 2px 6px; border: 1px solid #f97316; border-radius: 4px; background: #1a1a1f; color: #fff; font-size: 12px; outline: none;';

        el.replaceWith(input);
        input.focus();
        input.select();

        const saveArticle = () => {
          const newArticle = input.value.trim();
          collectedOrders[index].article = newArticle || null;
          renderOrdersPanel();
        };

        input.addEventListener('blur', saveArticle);
        input.addEventListener('keydown', (e) => {
          if (e.key === 'Enter') { e.preventDefault(); input.blur(); }
          if (e.key === 'Escape') { input.value = currentArticle; input.blur(); }
        });
      });
      el.style.cursor = 'pointer';
    });

    // Обработчики редактирования цвета
    listEl.querySelectorAll('.acreone-editable-color').forEach(el => {
      el.addEventListener('click', (e) => {
        e.stopPropagation();
        const index = parseInt(el.dataset.index);
        const order = collectedOrders[index];
        if (!order) return;

        const currentColor = order.color || '';
        const input = document.createElement('input');
        input.type = 'text';
        input.value = currentColor;
        input.className = 'acreone-color-input';
        input.placeholder = 'Цвет';
        input.style.cssText = 'width: 70px; padding: 2px 6px; border: 1px solid #f97316; border-radius: 4px; background: #1a1a1f; color: #fff; font-size: 12px; outline: none;';

        el.replaceWith(input);
        input.focus();
        input.select();

        const saveColor = () => {
          const newColor = input.value.trim();
          collectedOrders[index].color = newColor || null;
          renderOrdersPanel();
        };

        input.addEventListener('blur', saveColor);
        input.addEventListener('keydown', (e) => {
          if (e.key === 'Enter') { e.preventDefault(); input.blur(); }
          if (e.key === 'Escape') { input.value = currentColor; input.blur(); }
        });
      });
      el.style.cursor = 'pointer';
    });

    updateSendButton();

    // Обработчик отправки
    sendBtn.onclick = sendToAcreone;
  }

  // Обновление кнопки отправки
  function updateSendButton() {
    const selected = document.querySelectorAll('.acreone-order-select:checked').length;
    const btn = document.getElementById('acreone-send-btn');
    btn.textContent = `Отправить в AcreOne (${selected})`;
    btn.disabled = selected === 0;
  }

  // Прямой вызов API (минуя service worker для надёжности)
  async function importOrdersDirect(orders) {
    // Проверяем контекст расширения ПЕРЕД вызовом chrome API
    if (!isExtensionContextValid()) {
      throw new Error('EXTENSION_INVALID');
    }

    // Получаем токен и URL из storage
    let settings;
    try {
      settings = await chrome.storage.local.get(['acreone_api_url', 'acreone_token']);
    } catch (e) {
      if (e.message && e.message.includes('Extension context invalidated')) {
        throw new Error('EXTENSION_INVALID');
      }
      throw e;
    }
    const apiUrl = settings.acreone_api_url || 'https://api.acreone.ru';
    const token = settings.acreone_token;

    // ЗАЩИТА: Проверяем что URL - наш сервер (защита от подмены)
    const ALLOWED_API_HOSTS = ['api.acreone.ru', 'acreone.ru', 'localhost'];
    try {
      const urlObj = new URL(apiUrl);
      if (!ALLOWED_API_HOSTS.some(h => urlObj.hostname === h || urlObj.hostname.endsWith('.' + h))) {
        console.error('[AcreOne] Недоверенный API URL');
        throw new Error('Ошибка конфигурации. Переустановите расширение.');
      }
    } catch (e) {
      if (e.message.includes('конфигурации')) throw e;
      throw new Error('Неверный URL API');
    }

    if (!token) {
      throw new Error('Токен не настроен. Откройте настройки расширения.');
    }

    // SECURITY v2.0: Создаём защищённый запрос с подписью
    const secureRequest = await Security.createSecureRequest(orders, token);

    const response = await fetch(`${apiUrl}/rest/orders/import`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json',
        'X-AcreOne-Security-Version': '2.0'
      },
      body: JSON.stringify(secureRequest)
    });

    const result = await response.json();

    // Особая обработка лимита магазинов (402)
    if (response.status === 402) {
      return {
        success: false,
        shopLimitReached: true,
        error: result.message || 'Достигнут лимит магазинов',
        ...result
      };
    }

    if (!response.ok) {
      throw new Error(result.error || `Ошибка сервера: ${response.status}`);
    }

    return { success: true, ...result };
  }

  // Показать состояние отправки в панели
  function showSendingState(count) {
    const content = document.getElementById('acreone-orders-list');
    const sendBtn = document.getElementById('acreone-send-btn');

    if (sendBtn) sendBtn.style.display = 'none';

    content.innerHTML = `
      <div class="acreone-sending-state">
        <div class="acreone-spinner"></div>
        <h4>Отправка заказов...</h4>
        <p>Отправляем ${count} ${count === 1 ? 'заказ' : count < 5 ? 'заказа' : 'заказов'} в AcreOne</p>
      </div>
    `;
  }

  // Показать успешный результат
  function showSuccessState(imported, skipped) {
    const content = document.getElementById('acreone-orders-list');

    let skippedText = '';
    if (skipped > 0) {
      skippedText = `<p class="acreone-skipped">Пропущено (ошибки): <strong>${skipped}</strong></p>`;
    }

    const title = imported > 0 ? 'Заказы отправлены!' : 'Нет заказов для импорта';
    const importedText = imported > 0 ? `<p>Импортировано: <strong>${imported}</strong></p>` : '';

    content.innerHTML = `
      <div class="acreone-success-state">
        <div class="acreone-success-icon">
          <svg width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round">
            <polyline points="20 6 9 17 4 12"></polyline>
          </svg>
        </div>
        <h4>${title}</h4>
        ${importedText}
        ${skippedText}
        <a href="https://acreone.ru/manage/orders" target="_blank" class="acreone-goto-btn">
          Перейти в личный кабинет →
        </a>
        <button class="acreone-close-btn" onclick="document.getElementById('acreone-panel').classList.remove('open')">
          Закрыть
        </button>
      </div>
    `;
  }

  // Показать ошибку с возможностью повтора
  function showErrorState(errorMessage) {
    const content = document.getElementById('acreone-orders-list');
    // SECURITY: Экранируем сообщение об ошибке
    const safeMessage = escapeHtml(errorMessage);

    content.innerHTML = `
      <div class="acreone-error-state">
        <div class="acreone-error-icon">
          <svg width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round">
            <line x1="18" y1="6" x2="6" y2="18"></line>
            <line x1="6" y1="6" x2="18" y2="18"></line>
          </svg>
        </div>
        <h4>Ошибка отправки</h4>
        <p>${safeMessage}</p>
        <button class="acreone-retry-btn" id="acreone-retry-btn">
          Попробовать снова
        </button>
        <button class="acreone-close-btn" onclick="document.getElementById('acreone-panel').classList.remove('open')">
          Закрыть
        </button>
      </div>
    `;

    document.getElementById('acreone-retry-btn').onclick = () => {
      renderOrdersPanel();
    };
  }

  // Отправка в AcreOne
  async function sendToAcreone() {
    const selected = [];
    document.querySelectorAll('.acreone-order-select:checked').forEach(cb => {
      const index = parseInt(cb.dataset.index);
      const order = { ...collectedOrders[index] };

      // Применяем настройки сбора - удаляем отключённые поля
      if (collectSettings.collect_size === false) order.size = null;
      if (collectSettings.collect_article === false) order.article = null;
      if (collectSettings.collect_color === false) order.color = null;
      if (collectSettings.collect_buyer_name === false) order.buyer_name = null;
      if (collectSettings.collect_photo === false) {
        order.photo_url = null;
        order.photo_url_2 = null;
      }

      selected.push(order);
    });

    if (selected.length === 0) return;

    // Показываем состояние отправки
    showSendingState(selected.length);

    try {
      // Прямой вызов API (минуя service worker)
      const result = await importOrdersDirect(selected);

      if (result.shopLimitReached) {
        // Лимит магазинов достигнут
        showShopLimitError(result);
        return;
      }

      if (!result.success) {
        throw new Error(result.error || 'Неизвестная ошибка');
      }

      // Успешно - показываем результат
      showSuccessState(result.imported || 0, result.skipped || 0);

      // Обновляем статистику в storage (с защитой от невалидного контекста)
      try {
        if (isExtensionContextValid()) {
          const stats = await chrome.storage.local.get(['stats_collected']);
          await chrome.storage.local.set({
            stats_collected: (stats.stats_collected || 0) + selected.length
          });
        }
      } catch (statsErr) {
        // Игнорируем ошибки статистики - заказы уже отправлены
      }

    } catch (error) {
      console.error('[AcreOne] Ошибка отправки:', error);

      // Специальная обработка невалидного контекста расширения
      if (error.message === 'EXTENSION_INVALID' ||
          (error.message && error.message.includes('Extension context invalidated'))) {
        showExtensionInvalidError();
        return;
      }

      showErrorState(error.message);
    }
  }

  // Показ ошибки лимита магазинов
  function showShopLimitError(result) {
    const panel = document.getElementById('acreone-panel');
    const content = document.getElementById('acreone-orders-list');
    // SECURITY: Экранируем название магазина
    const safeShopName = escapeHtml(result.currentShops?.[0]?.name || 'Неизвестный');

    content.innerHTML = `
      <div class="acreone-limit-error">
        <div class="acreone-limit-icon">
          <svg width="56" height="56" viewBox="0 0 24 24" fill="none" stroke="#fbbf24" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"></path>
            <line x1="12" y1="9" x2="12" y2="13"></line>
            <line x1="12" y1="17" x2="12.01" y2="17"></line>
          </svg>
        </div>
        <h4>Достигнут лимит магазинов</h4>
        <p>Ваш тариф позволяет работать только с одним магазином.</p>
        <p>Текущий магазин: <strong>${safeShopName}</strong></p>
        <p style="margin-top: 12px;">Для работы с несколькими магазинами обновите тариф:</p>
        <a href="https://acreone.ru/manage/billing" target="_blank" class="acreone-upgrade-btn">
          Обновить тариф
        </a>
      </div>
    `;

    // Скрываем кнопку отправки
    document.getElementById('acreone-send-btn').style.display = 'none';
  }

  // Показ статуса
  function showStatus(message, type = '') {
    let statusEl = document.querySelector('.acreone-status');
    if (!statusEl) {
      statusEl = document.createElement('div');
      statusEl.className = 'acreone-status';
      document.body.appendChild(statusEl);
    }

    statusEl.textContent = message;
    statusEl.className = 'acreone-status ' + type;

    // Автоскрытие успешных сообщений
    if (type === 'success') {
      setTimeout(() => {
        statusEl.remove();
      }, 3000);
    }
  }

  // Утилиты
  function sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }

  // Запуск
  init();
})();
