/**
 * AcreOne - Popup script
 * v1.10.25 - Добавлена страница настроек сбора при первом запуске
 */

document.addEventListener('DOMContentLoaded', async () => {
  // Загружаем ВСЕ сохранённые настройки
  const settings = await chrome.storage.local.get([
    'acreone_api_url',
    'acreone_token',
    'stats_collected',
    'stats_sent',
    'collect_size',
    'collect_article',
    'collect_color',
    'collect_buyer_name',
    'collect_photo',
    'collect_name',
    'photo_count',
    'size_source',
    'allowed_shops',
    'max_shops',
    'extension_enabled',
    'shops_expanded',
    'onboarding_completed',
    'binding_confirmed',
    'collect_settings_done',
    'rebind_days'
  ]);

  console.log('[AcreOne Popup] Loaded settings:', settings);

  // === TOGGLE РАСШИРЕНИЯ ===
  const toggle = document.getElementById('extension-toggle');
  const toggleStatus = document.getElementById('toggle-status');
  const isEnabled = settings.extension_enabled !== false;

  function updateToggleUI(enabled) {
    if (enabled) {
      toggle.classList.add('active');
      toggleStatus.textContent = 'Включено';
      toggleStatus.classList.add('active');
    } else {
      toggle.classList.remove('active');
      toggleStatus.textContent = 'Выключено';
      toggleStatus.classList.remove('active');
    }
  }

  updateToggleUI(isEnabled);

  toggle.addEventListener('click', async () => {
    const newState = !toggle.classList.contains('active');
    await chrome.storage.local.set({ extension_enabled: newState });
    updateToggleUI(newState);

    // Уведомляем content script
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab?.url?.includes('avito.ru')) {
        chrome.tabs.sendMessage(tab.id, { type: 'TOGGLE_EXTENSION', enabled: newState });
      }
    } catch (e) {}
  });

  // === ДИНАМИЧЕСКИЙ UI ===
  const onboarding = document.getElementById('onboarding');
  const bindingSection = document.getElementById('binding-section');
  const collectSettingsSection = document.getElementById('collect-settings-section');
  const mainContent = document.getElementById('main-content');

  // Проверяем есть ли токен
  if (settings.acreone_token) {
    // Токен есть
    if (!settings.binding_confirmed) {
      // Показываем предупреждение о привязке
      showBindingWarning(settings.rebind_days || 30);
    } else if (!settings.collect_settings_done) {
      // Показываем настройки сбора
      showCollectSettings(settings);
    } else {
      // Всё подтверждено - показываем основной контент
      showMainContent(settings);
    }
  } else {
    // Токена нет - показываем онбординг
    showOnboarding();
  }

  // === ОНБОРДИНГ ===
  const onboardingNext1 = document.getElementById('onboarding-next-1');
  const onboardingBack = document.getElementById('onboarding-back');

  onboardingNext1?.addEventListener('click', () => {
    // Переключаем на страницу 2
    document.querySelector('.onboarding-page[data-page="1"]')?.classList.remove('active');
    document.querySelector('.onboarding-page[data-page="2"]')?.classList.add('active');
  });

  onboardingBack?.addEventListener('click', () => {
    // Возврат на страницу 1
    document.querySelector('.onboarding-page[data-page="2"]')?.classList.remove('active');
    document.querySelector('.onboarding-page[data-page="1"]')?.classList.add('active');
  });

  // === ПРИВЯЗКА ===
  document.getElementById('confirm-binding')?.addEventListener('click', async () => {
    await chrome.storage.local.set({ binding_confirmed: true });
    showCollectSettings(settings);
  });

  // === НАСТРОЙКИ СБОРА (первый запуск) ===
  document.getElementById('save-collect-settings')?.addEventListener('click', async () => {
    // Собираем настройки с формы
    const photoCount = document.querySelector('input[name="photo-count"]:checked')?.value || '1';
    const collectName = document.getElementById('setup-collect-name')?.checked ?? true;
    const collectColor = document.getElementById('setup-collect-color')?.checked ?? true;
    const sizeSource = document.querySelector('input[name="size-source"]:checked')?.value || 'ai';
    const collectArticle = document.getElementById('setup-collect-article')?.checked ?? true;

    // Преобразуем в старые настройки для совместимости
    const collectSettings = {
      photo_count: parseInt(photoCount),
      collect_photo: photoCount !== '0',
      collect_name: collectName,
      collect_color: collectColor,
      size_source: sizeSource,
      collect_size: sizeSource !== 'none',
      collect_article: collectArticle,
      collect_buyer_name: true, // Всегда собираем
      collect_settings_done: true
    };

    // Сохраняем локально
    await chrome.storage.local.set(collectSettings);

    // Пытаемся сохранить на сервер
    try {
      const { acreone_api_url, acreone_token } = await chrome.storage.local.get([
        'acreone_api_url', 'acreone_token'
      ]);

      if (acreone_token) {
        await fetch(`${acreone_api_url}/rest/orders/settings`, {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${acreone_token}`,
            'Content-Type': 'application/json'
          },
          body: JSON.stringify(collectSettings)
        });
      }
    } catch (e) {
      console.error('[AcreOne] Ошибка сохранения настроек на сервер:', e);
    }

    // Переходим к основному контенту
    showMainContent(Object.assign(settings, collectSettings));
  });

  // === ОБРАБОТЧИКИ ===

  // Сохранить токен
  document.getElementById('save-btn').addEventListener('click', async () => {
    const apiUrl = 'https://api.acreone.ru';
    const token = document.getElementById('api-token').value.trim();

    if (!token) {
      showStatus('Введите токен авторизации', 'error');
      return;
    }

    showStatus('Проверка токена...', '');

    try {
      const response = await fetch(`${apiUrl}/rest/accounts`, {
        headers: { 'Authorization': `Bearer ${token}` }
      });

      if (response.ok) {
        // Получаем данные о тарифе для rebind_days
        let rebindDays = 30;
        try {
          const settingsResp = await fetch(`${apiUrl}/rest/orders/settings`, {
            headers: { 'Authorization': `Bearer ${token}` }
          });
          if (settingsResp.ok) {
            const data = await settingsResp.json();
            rebindDays = data.rebind_days || 30;
          }
        } catch (e) {}

        // Токен валидный - сохраняем
        await chrome.storage.local.set({
          acreone_api_url: apiUrl,
          acreone_token: token,
          rebind_days: rebindDays,
          onboarding_completed: true
        });

        showStatus('Подключено!', 'success');

        // Показываем предупреждение о привязке
        setTimeout(() => {
          showBindingWarning(rebindDays);
        }, 500);

      } else if (response.status === 401 || response.status === 403) {
        showStatus('Неверный токен', 'error');
      } else {
        showStatus(`Ошибка: ${response.status}`, 'error');
      }
    } catch (error) {
      showStatus('Ошибка подключения', 'error');
    }
  });

  // Выход (сброс токена)
  document.getElementById('logout-btn')?.addEventListener('click', async () => {
    await chrome.storage.local.remove(['acreone_token']);
    showAuthOnly();
  });

  // === СВОРАЧИВАЕМЫЕ МАГАЗИНЫ ===
  const shopsHeader = document.getElementById('shops-header');
  const shopsList = document.getElementById('shops-list');
  const shopsToggleIcon = document.getElementById('shops-toggle-icon');

  // Восстанавливаем состояние
  if (settings.shops_expanded) {
    shopsList?.classList.add('expanded');
    shopsToggleIcon?.classList.add('expanded');
  }

  shopsHeader?.addEventListener('click', async () => {
    const isExpanded = shopsList?.classList.toggle('expanded');
    shopsToggleIcon?.classList.toggle('expanded');
    await chrome.storage.local.set({ shops_expanded: isExpanded });
  });

  // === КНОПКА НАСТРОЕК (показывает страницу настроек) ===
  const openSettingsBtn = document.getElementById('open-settings-btn');
  openSettingsBtn?.addEventListener('click', async () => {
    // Загружаем текущие настройки
    const currentSettings = await chrome.storage.local.get([
      'photo_count', 'collect_name', 'collect_color', 'size_source', 'collect_article'
    ]);
    showCollectSettings(currentSettings, true); // true = показать кнопку "Назад"
  });

  // === КНОПКА НАЗАД ИЗ НАСТРОЕК ===
  document.getElementById('back-from-settings')?.addEventListener('click', async () => {
    const currentSettings = await chrome.storage.local.get([
      'acreone_api_url', 'acreone_token', 'stats_collected', 'stats_sent',
      'collect_size', 'collect_article', 'collect_color', 'collect_buyer_name',
      'collect_photo', 'collect_name', 'photo_count', 'size_source',
      'allowed_shops', 'max_shops'
    ]);
    showMainContent(currentSettings);
  });
});

// Показать онбординг
function showOnboarding() {
  document.getElementById('onboarding').classList.add('show');
  document.getElementById('binding-section').style.display = 'none';
  document.getElementById('collect-settings-section').style.display = 'none';
  document.getElementById('main-content').style.display = 'none';
  document.getElementById('toggle-section').style.display = 'none';
  // Сбрасываем на первую страницу
  document.querySelectorAll('.onboarding-page').forEach(p => p.classList.remove('active'));
  document.querySelector('.onboarding-page[data-page="1"]')?.classList.add('active');
}

// Показать предупреждение о привязке
function showBindingWarning(days) {
  document.getElementById('onboarding').classList.remove('show');
  document.getElementById('binding-section').style.display = 'block';
  document.getElementById('collect-settings-section').style.display = 'none';
  document.getElementById('main-content').style.display = 'none';
  document.getElementById('toggle-section').style.display = 'none';
  document.getElementById('rebind-days').textContent = days;
}

// Показать страницу настроек сбора
// showBackButton = true если открыли из главного экрана (не из онбординга)
function showCollectSettings(settings, showBackButton = false) {
  document.getElementById('onboarding').classList.remove('show');
  document.getElementById('binding-section').style.display = 'none';
  document.getElementById('collect-settings-section').style.display = 'block';
  document.getElementById('main-content').style.display = 'none';
  document.getElementById('toggle-section').style.display = 'none';

  // Показываем/скрываем кнопку "Назад"
  const backBtn = document.getElementById('back-from-settings');
  if (backBtn) {
    backBtn.style.display = showBackButton ? 'block' : 'none';
  }

  // Восстанавливаем сохранённые настройки если есть
  const photoCount = settings.photo_count ?? 1;
  const photoRadio = document.querySelector(`input[name="photo-count"][value="${photoCount}"]`);
  if (photoRadio) photoRadio.checked = true;

  const setupName = document.getElementById('setup-collect-name');
  if (setupName) setupName.checked = settings.collect_name !== false;

  const setupColor = document.getElementById('setup-collect-color');
  if (setupColor) setupColor.checked = settings.collect_color !== false;

  const sizeSource = settings.size_source || 'ai';
  const sizeRadio = document.querySelector(`input[name="size-source"][value="${sizeSource}"]`);
  if (sizeRadio) sizeRadio.checked = true;

  const setupArticle = document.getElementById('setup-collect-article');
  if (setupArticle) setupArticle.checked = settings.collect_article !== false;
}

// Показать только форму авторизации (для кнопки "Сменить токен")
function showAuthOnly() {
  showOnboarding();
  // Переключаем сразу на страницу ввода токена
  document.querySelector('.onboarding-page[data-page="1"]')?.classList.remove('active');
  document.querySelector('.onboarding-page[data-page="2"]')?.classList.add('active');
  document.getElementById('api-token').value = '';
}

// Показать основной контент
function showMainContent(settings) {
  document.getElementById('onboarding').classList.remove('show');
  document.getElementById('binding-section').style.display = 'none';
  document.getElementById('collect-settings-section').style.display = 'none';
  document.getElementById('main-content').style.display = 'block';
  document.getElementById('toggle-section').style.display = 'flex';

  // Статистика
  document.getElementById('stat-collected').textContent = settings.stats_collected || 0;
  document.getElementById('stat-sent').textContent = settings.stats_sent || 0;

  // Магазины
  renderShops(settings.allowed_shops || [], settings.max_shops || 1);

  // Загружаем актуальные настройки с сервера
  loadServerSettings();

  // Загружаем месячную статистику
  loadMonthlyStats();
}

// Загрузка настроек с сервера
async function loadServerSettings() {
  try {
    console.log('[AcreOne Popup] Запрос настроек с сервера...');
    const response = await chrome.runtime.sendMessage({ type: 'FETCH_SERVER_SETTINGS' });
    console.log('[AcreOne Popup] Ответ от сервера:', response);

    if (response?.success) {
      const s = response.settings;
      console.log('[AcreOne Popup] Магазины:', s.allowed_shops);

      // Сохраняем локально
      await chrome.storage.local.set({
        collect_size: s.collect_size,
        collect_article: s.collect_article,
        collect_color: s.collect_color,
        collect_buyer_name: s.collect_buyer_name,
        collect_photo: s.collect_photo,
        photo_count: s.photo_count,
        size_source: s.size_source,
        collect_name: s.collect_name
      });

      // Магазины
      renderShops(s.allowed_shops || [], s.max_shops || 1);
    } else {
      console.error('[AcreOne Popup] Ошибка получения настроек:', response?.error || 'Unknown error');
    }
  } catch (error) {
    console.error('[AcreOne Popup] Exception loading settings:', error);
  }
}

// Отображение магазинов (компактный вид)
function renderShops(shops, maxShops) {
  const listEl = document.getElementById('shops-list');
  const infoEl = document.getElementById('shop-limit-info');

  if (!listEl) return;

  if (shops.length === 0) {
    listEl.innerHTML = '<p class="shops-empty">Магазины добавятся автоматически при первом сборе заказов</p>';
  } else {
    listEl.innerHTML = shops.map(shop => `
      <div class="shop-item">
        <span class="shop-name" title="${shop.avito_shop_name || 'Магазин'}">${shop.avito_shop_name || 'Магазин'}</span>
        <span class="shop-id">${shop.avito_user_id}</span>
      </div>
    `).join('');
  }

  if (infoEl) {
    if (shops.length >= maxShops) {
      infoEl.className = 'shop-limit-info';
      infoEl.innerHTML = `⚠️ Лимит: ${shops.length}/${maxShops}`;
    } else {
      infoEl.className = 'shop-limit-info ok';
      infoEl.innerHTML = `✓ ${shops.length}/${maxShops}`;
    }
  }
}

function showStatus(message, type, targetId = null) {
  // Определяем где показывать статус
  let statusEl;
  if (targetId) {
    statusEl = document.getElementById(targetId);
  } else {
    // Показываем в видимой секции
    const mainContent = document.getElementById('main-content');
    if (mainContent && mainContent.style.display !== 'none') {
      statusEl = document.getElementById('main-status');
    } else {
      statusEl = document.getElementById('status');
    }
  }

  if (!statusEl) return;

  statusEl.textContent = message;
  statusEl.className = 'status ' + (type || '');
  statusEl.style.display = 'block';

  if (type === 'success') {
    setTimeout(() => {
      statusEl.className = 'status';
      statusEl.style.display = 'none';
    }, 3000);
  }
}

// Загрузка месячной статистики с сервера
async function loadMonthlyStats() {
  const refreshBtn = document.getElementById('refresh-stats-btn');
  const refreshText = document.getElementById('refresh-stats-text');

  try {
    if (refreshText) refreshText.textContent = 'Загрузка...';
    if (refreshBtn) refreshBtn.disabled = true;

    const { acreone_api_url, acreone_token } = await chrome.storage.local.get([
      'acreone_api_url', 'acreone_token'
    ]);

    if (!acreone_token) {
      console.log('[AcreOne Popup] Нет токена для загрузки статистики');
      return;
    }

    const response = await fetch(`${acreone_api_url}/rest/orders/stats/monthly`, {
      headers: { 'Authorization': `Bearer ${acreone_token}` }
    });

    if (response.ok) {
      const stats = await response.json();
      console.log('[AcreOne Popup] Месячная статистика:', stats);

      // Обновляем UI
      document.getElementById('stat-month-orders').textContent = stats.orders_count || 0;
      document.getElementById('stat-month-items').textContent = stats.items_count || 0;

      // Форматируем сумму
      const revenue = stats.total_revenue || 0;
      const formattedRevenue = revenue.toLocaleString('ru-RU') + ' ₽';
      document.getElementById('stat-month-revenue').textContent = formattedRevenue;
    } else {
      console.error('[AcreOne Popup] Ошибка загрузки статистики:', response.status);
      document.getElementById('stat-month-orders').textContent = '—';
      document.getElementById('stat-month-items').textContent = '—';
      document.getElementById('stat-month-revenue').textContent = '—';
    }
  } catch (error) {
    console.error('[AcreOne Popup] Ошибка загрузки месячной статистики:', error);
  } finally {
    if (refreshText) refreshText.textContent = 'Обновить';
    if (refreshBtn) refreshBtn.disabled = false;
  }
}

// Инициализация обработчика кнопки обновления статистики
document.getElementById('refresh-stats-btn')?.addEventListener('click', loadMonthlyStats);
