/**
 * AcreOne Security Module v2.1.0
 * Защитные механизмы для расширения
 */

(function() {
  'use strict';

  // Секретный ключ для HMAC (уникальный для v2)
  // ВАЖНО: Этот ключ видим в коде, он лишь усложняет атаку
  const HMAC_KEY = 'acr30n3-3xt-v2-s3cur1ty-2024';

  // SHA-256 реализация для браузера
  async function sha256(message) {
    const msgBuffer = new TextEncoder().encode(message);
    const hashBuffer = await crypto.subtle.digest('SHA-256', msgBuffer);
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
  }

  // HMAC-SHA256 для браузера
  async function hmacSha256(message, key) {
    const encoder = new TextEncoder();
    const keyData = encoder.encode(key);
    const messageData = encoder.encode(message);

    const cryptoKey = await crypto.subtle.importKey(
      'raw',
      keyData,
      { name: 'HMAC', hash: 'SHA-256' },
      false,
      ['sign']
    );

    const signature = await crypto.subtle.sign('HMAC', cryptoKey, messageData);
    const hashArray = Array.from(new Uint8Array(signature));
    return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
  }

  // Генерация случайного nonce
  function generateNonce() {
    const array = new Uint8Array(16);
    crypto.getRandomValues(array);
    return Array.from(array).map(b => b.toString(16).padStart(2, '0')).join('');
  }

  // Хеширование данных заказов (для integrity check)
  async function hashOrdersData(orders) {
    // Нормализуем данные для детерминированного хеша
    const normalized = orders.map(o => ({
      avito_order_id: String(o.avito_order_id || ''),
      avito_user_id: String(o.avito_user_id || ''),
      price: Number(o.price) || 0,
      track_number: String(o.track_number || '')
    }));

    // Сортируем для детерминированности
    normalized.sort((a, b) => a.avito_order_id.localeCompare(b.avito_order_id));

    const hash = await sha256(JSON.stringify(normalized));
    return hash.substring(0, 16); // Короткий хеш достаточен
  }

  // Создание защищённого запроса
  async function createSecureRequest(orders, token) {
    const timestamp = Date.now();
    const nonce = generateNonce();
    const dataHash = await hashOrdersData(orders);

    // Подпись: token + timestamp + nonce + dataHash
    const signaturePayload = `${token}:${timestamp}:${nonce}:${dataHash}`;
    const signature = await hmacSha256(signaturePayload, HMAC_KEY);

    return {
      orders,
      _security: {
        version: '2.0',
        timestamp,
        nonce,
        dataHash,
        signature
      }
    };
  }

  // Заморозка объекта (защита от модификации)
  function freezeDeep(obj) {
    if (obj && typeof obj === 'object') {
      Object.keys(obj).forEach(key => freezeDeep(obj[key]));
      return Object.freeze(obj);
    }
    return obj;
  }

  // Защищённое хранилище для collectedOrders
  function createSecureOrdersStorage() {
    let _orders = [];
    let _hash = null;
    let _frozen = false;

    return {
      push(order) {
        if (_frozen) {
          console.warn('[AcreOne Security] Попытка модификации замороженных данных');
          return false;
        }
        _orders.push(freezeDeep({ ...order }));
        _hash = null; // Инвалидируем хеш
        return true;
      },

      get(index) {
        return _orders[index];
      },

      getAll() {
        return [..._orders]; // Возвращаем копию
      },

      get length() {
        return _orders.length;
      },

      clear() {
        _orders = [];
        _hash = null;
        _frozen = false;
      },

      freeze() {
        _frozen = true;
      },

      async getHash() {
        if (!_hash) {
          _hash = await hashOrdersData(_orders);
        }
        return _hash;
      },

      // Проверка целостности
      async verifyIntegrity(expectedHash) {
        const currentHash = await this.getHash();
        return currentHash === expectedHash;
      },

      // Итератор для forEach
      forEach(callback) {
        _orders.forEach((order, index) => callback({ ...order }, index));
      },

      map(callback) {
        return _orders.map((order, index) => callback({ ...order }, index));
      },

      filter(callback) {
        return _orders.filter((order, index) => callback({ ...order }, index));
      }
    };
  }

  // Экспорт в глобальный объект (защищённый)
  const AcreOneSecurity = Object.freeze({
    createSecureRequest,
    createSecureOrdersStorage,
    hashOrdersData,
    generateNonce,
    freezeDeep,
    VERSION: '2.1'
  });

  // Защищаем от переопределения
  Object.defineProperty(window, '__AcreOneSecurity', {
    value: AcreOneSecurity,
    writable: false,
    configurable: false,
    enumerable: false
  });

})();
